<?php

namespace App\Model;

use App\Entity\Fornecedor;
use App\Entity\Grupo;
use App\Entity\Inventario;
use App\Entity\InventarioEquipe;
use App\Entity\InventarioProdutoTmp;
use App\Entity\InventarioLote;
use App\Entity\InventarioLoteTmp;
use App\Entity\ItensMovimento;
use App\Entity\Movimento;
use App\Entity\Produto;
use App\Entity\Setor;
use App\Entity\Usuario;
use App\Helper\Filtros;
use App\Helper\MovEntrada;
use App\Helper\MovSaida;
use App\Repository\MovimentacaoRepo;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Query\ResultSetMapping;
use Exception;
use DateTime;
use Saude\Horus\Sync\SyncController;

class MovimentoModel extends AbstractModel implements MovimentacaoRepo {

	use Filtros;
	const USER_AUTH = "Usuário não autorizado.";
	const MOV_OBS = "MOVIMENTACAO DE INVENTARIO";

	public function __construct() {
		parent::__construct();
	}

	public function abrirInventario($dados) {
		try {
			$criteria = [
				"data" => new DateTime($dados['data']),
				"grupo" => $dados['gru_codigo'],
				"setor" => $dados['set_codigo'],
				"status" => Inventario::ABERTO
			];

			$inventario = $this->_getObject(Inventario::class, $criteria);
			if (count($inventario) > 0) {
				return reset($inventario)->getId();
			}

			$inventario = new Inventario();
			$inventario->setData(new DateTime($dados['data']));
			$inventario->setSetor($this->_getOneObject(Setor::class, $dados['set_codigo']));
			$inventario->setGrupo($this->_getOneObject(Grupo::class, $dados['gru_codigo']));
			$inventario->setResponsavel($dados['responsavel']);
			$inventario->setEquipe($dados['equipe']);
			$inventario->setUsuario($this->_getOneObject(Usuario::class, $dados['usr_codigo']));
			$inventario->setDataDigitacao(new DateTime(date('y-m-d')));
			$inventario->setIp($_SERVER['REMOTE_ADDR']);
			$inventario->setStatus(Inventario::ABERTO);

			return $this->findInventarioById($this->salvar($inventario));
		} catch (Exception $e) {
			die($e->getMessage());
		}
	}

	public function findInventarioById($inv_codigo) {
		try {
			$inventario = $this->_getOneObject(Inventario::class, $inv_codigo);
			if (isset($inventario)) {
				$equipe = $this->_getObject(InventarioEquipe::class, ['inventario' => $inventario->getId()]);
				$inventario->setEquipe(
					array_map(function (InventarioEquipe $eq) {
						$eq = $eq->toArray();
						unset($eq['Inventario']);
						return $eq;
					}, $equipe)
				);
			}

			return isset($inventario) ? $inventario->toArray() : $inventario;
		} catch (Exception $e) {
			die("Erro: " . $e->getMessage());
		}
	}

	public function findAllInventario() {
		try {
			$inventario = $this->getEm()->getRepository(Inventario::class)->findAll();
			$inventario = array_map(function (Inventario $inv) {
				$equipe = $this->_getObject(InventarioEquipe::class, ["inventario" => $inv->getId()]);
				$inv->setEquipe(self::_arrayMap($equipe));
				return $inv->toArray();
			}, $inventario);

			return $inventario;
		} catch (\Exception $e) {
			die($e->getMessage());
		}
	}

	public function carregaProdutosInventario($dados) {
		try {
			$inventario = $this->_getOneObject(Inventario::class, $dados['inv_codigo']);

			if ($inventario->getStatus() == Inventario::ABERTO) {
				$inv_produtos = $this->_getObject(InventarioProdutoTmp::class, [
					"inventario" => $dados['inv_codigo']
				]);

				if (count($inv_produtos) > 0) {
					return self::_arrayMap($inv_produtos);
				} else {
					$setor = $inventario->getSetor();
					$grupo = $inventario->getGrupo();
					$itens = $this->getAllProdutoInventario($grupo['id'], $setor['id']);

					if (count($itens) > 0) {
						foreach ($itens as $item) {
							$produto = $this->_getOneObject(Produto::class, $item['pro_codigo']);
							$usuario = $this->_getOneObject(Usuario::class, $dados['usr_codigo']);
							$inv_pro = new InventarioProdutoTmp();
							$inv_pro->setInventario($inventario);
							$inv_pro->setProduto($produto);
							$inv_pro->setUsuario($usuario);
							$inv_pro->setLote($item['ite_lote']);
							$inv_pro->setQuantidade($item['sal_qtde']);
							$inv_pro->setValidade(new DateTime($item['ite_validade']));
							$inv_pro->setData(new DateTime());
							$inv_pro->setStatus(InventarioProdutoTmp::ATIVO);
							$inv_pro->setIp($_SERVER["REMOTE_ADDR"]);
							$inv_pro->setDoseAtual($item['sal_dose_atual']);
							$inv_pro->setDoseLote($item['sal_dose_lote']);

							$this->salvar($inv_pro);
						}
					}

					return self::_arrayMap($this->_getObject(InventarioProdutoTmp::class, [
						"inventario" => $dados['inv_codigo']
					]));
				}
			} else {
				return "Este inventário já foi processado. Abra um novo inventário para realizar este procedimento";
			}
		} catch (Exception $e) {
			die($e->getMessage());
		}
	}

	public function alteraProdutoInventario($dados) {
		try {
			$inv_pro = $this->_getOneObject(InventarioProdutoTmp::class, $dados["id"]);
			if (!isset($inv_pro)) {
				return null;
			}

			$inv_pro->setData(new DateTime());
			$inv_pro->setQuantidade($dados['quantidade']);
			$inv_pro->setIp($_SERVER['REMOTE_ADDR']);
			$inv_pro->setStatus(InventarioProdutoTmp::ALTERADO);
			$inv_pro->setUsuario($this->_getOneObject(Usuario::class, $dados["usuario"]));

			if($this->salvar($inv_pro)){
				return $inv_pro->toArray();
			}
		} catch (Exception $e) {
			return $e->getMessage();
		}
	}

	public function abrirLote($dados) {
		try {
			$inventarioLoteTmp = new InventarioLoteTmp();
			$inventarioLoteTmp->setInventarioProduto($this->_getOneObject(InventarioProdutoTmp::class, $dados['inventario_produto']));

			return $this->salvar($inventarioLoteTmp);
		} catch (Exception $e) {
			die($e->getMessage());
		}
	}

	public function alteraLote($dados) {
		try {
			$class = ($dados['tipo'] == 'temporario' ? InventarioLoteTmp::class : InventarioLote::class);

			$inventarioLote = $this->_getOneObject($class, $dados["id"]);
			if (!isset($inventarioLote)) {
				return null;
			}

			$inventarioLote->setLote($dados['lote']);
			$inventarioLote->setValidade(new DateTime($dados['validade']));
			$inventarioLote->setQuantidade($dados['quantidade']);
			$inventarioLote->setQuantidadeSaldo($dados['quantidade_saldo']);
			$inventarioLote->setDoseLote($dados['dose_lote']);
			$inventarioLote->setDoseAtual($dados['dose_atual']);

			return $this->salvar($inventarioLote);
		} catch (Exception $e) {
			return $e->getMessage();
		}
	}

	public function removeLote($dados) {
		try {
			$lote = $this->_getOneObject(InventarioLoteTmp::class, $dados["invplq_codigo"]);
			if (!isset($lote)) {
				return 'Nenhum lote encontrado!';
			}

			$this->deletar(InventarioLoteTmp::class, $dados['invplq_codigo']);

			return true;
		} catch (Exception $e) {
			return $e->getMessage();
		}
	}

	public function findLoteByProduct($invp_codigo) {
		try {
			$lote = $this->_getObject(InventarioLoteTmp::class, [
				"inventario_produto" => $invp_codigo,
			]);
			if (isset($lote)) {
				return array_map(function (InventarioLoteTmp $lot) {
					return $lot->toArray();
				}, $lote);
			}

			return NULL;
		} catch (Exception $e) {
			die("Erro: " . $e->getMessage());
		}
	}

	public function findAllLote() {
		try {
			$lote = $this->getEm()->getRepository(InventarioLoteTmp::class)->findAll();

			return array_map(function (InventarioLoteTmp $lot) {
				return $lot->toArray();
			}, $lote);
		} catch (Exception $e) {
			die($e->getMessage());
		}
	}

	public function cancelaInventario($inv_codigo) {
		try {
			$inventarioProdutoTmp = self::_arrayMap($this->_getObject(InventarioProdutoTmp::class, [
				"inventario" => $inv_codigo,
			]));
			foreach ($inventarioProdutoTmp as $key => $invProd) {
				$arr[] = $invProd['id'];
			}
			if (count($arr) > 0) {
				$inventarioLoteTmp = self::_arrayMap($this->_getObject(InventarioLoteTmp::class, [
					"inventario_produto" => $arr
				]));
				return $this->excluiRegistrosInventario($inv_codigo, $inventarioProdutoTmp, $inventarioLoteTmp);
			}

			return $this->excluiRegistrosInventario($inv_codigo, $inventarioProdutoTmp);
		} catch (Exception $e) {
			return $e->getMessage();
		}
	}

	/**
	*  PROCESSO
	*  1 - Selecionar todos produtos com alteração
	*  2 - Processar no Horus e aguardar retorno
	*  3 - Em retorno positivo faz a movimentação do produto
	*  4 - Salva os produtos na tabela inventario produto
	*  5 - Altera os campos na tabela Inventario (inv_status = F, inv_data_digitacao = new Datetime())
	*/
	public function processarInventario($inv_codigo) {
		try {
			$repo = $this->_getObject(InventarioProdutoTmp::class, [
				"inventario" => $inv_codigo,
			]);

			$retorno = [];
			if (count($repo) > 0) {
				$dtAtual = new DateTime();

				//Erros do Horus
				$err_server_horus = [];

				//Erros dos Produtos
				$err_pro_horus = [];

				//Fornecedor tipo INVENTÁRIO
				$fornecedor = $this->_getOneObjectBy(Fornecedor::class, [
					"for_nome" => "INVENTARIO"
				]);
				if (!isset($fornecedor)) {
					return "Fornecedor 'INVENTARIO' não foi cadastrado.";
				}
				//Configurações
				$config = $this->getConfig([
					'CID_CODIGO_IBGE',
					'VALIDADE_RECEITA'
				]);

				//Inventário
				$inventario = $this->_getOneObject(Inventario::class, $inv_codigo);
				if (!isset($inventario)) {
					return "Inventário não encontrado";
				}

				//Setor do Inventário
				$setor = $this->_getOneObject(Setor::class, $inventario->getSetor()['id']);

				$horus = new SyncController();
				$itensMovimentoEntrada = [];
				$itensMovimentoSaida = [];
				foreach (self::_arrayMap($repo) as $produto) {
					if (empty(trim($produto['produto']['proHorus']))) {
						$err_pro_horus[$produto['produto']['id']] = $produto['produto']['proNome'];
					}
					else {
						$protocolo = $horus->informarPosicaoEstoque(
							null,
							null,
							null,
							null,
							null,
							$produto['produto']['proHorus'],
							$produto['lote'],
							date('d-m-Y', strtotime($produto['validade'])),
							$produto['quantidade'],
							$dtAtual->format('d-m-Y')
						);
						if ($protocolo instanceof \SoapFault) {
							if ($protocolo->getMessage() != $this::USER_AUTH) {
								$err_server_horus[$produto['id']] = $protocolo->getMessage();
							}
						} else {
							//Grava o protocolo
							$retorno['protocolos'][] = $protocolo;
						}
					}

					if ($produto['quantidade'] < $produto['quantidadeOriginal']) {
						$itensMovimentoSaida[] = $produto;
					} else {
						$itensMovimentoEntrada[] = $produto;
					}
				}

				//Movimento de Saída
				if (count($itensMovimentoSaida) > 0) {
					$mov_saida = new Movimento();
					$mov_saida->setMovTipo(Movimento::SAIDA);
					$mov_saida->setMovSaida(MovSaida::SAIDA_INVENTARIO);
					$mov_saida->setMovData(new DateTime());
					$mov_saida->setMovDataInclusao(new DateTime());
					$mov_saida->setMovDataInsercao(new DateTime());
					$mov_saida->setInventario($inventario);
					$mov_saida->setFornecedor($fornecedor);
					$mov_saida->setSetorSaida($setor);
					$mov_saida->setMovObservacao($this::MOV_OBS);
					$this->salvar($mov_saida);

					foreach ($itensMovimentoSaida as $item) {
						$item_mov_saida = new ItensMovimento();
						$item_mov_saida->setMovimento($mov_saida);
						$item_mov_saida->setProduto($this->_getOneObject(Produto::class, $item['produto']['id']));
						$item_mov_saida->setIteLote($item['lote']);
						$item_mov_saida->setIteValidade(new DateTime($item['validade']));
						$item_mov_saida->setIteStatus(ItensMovimento::FECHADO);
						$item_mov_saida->setIteQuantidade(abs($item['quantidade'] - $item['quantidadeOriginal']));
						$item_mov_saida->setIteConsolidado(ItensMovimento::CONSOLIDADO_SIM);
						$item_mov_saida->setIteDose($item['doseAtual']);
						$item_mov_saida->setIteDoseLote($item['doseLote']);

						if ($this->salvar($item_mov_saida)) {
							$this->deletar(InventarioProdutoTmp::class, $item['id']);
						}
					}
				}

				//Movimento de Entrada
				if (count($itensMovimentoEntrada) > 0) {
					$mov_entrada = new Movimento();
					$mov_entrada->setMovTipo(Movimento::ENTRADA);
					$mov_entrada->setMovEntrada(MovEntrada::ENTRADA_INVENTARIO);
					$mov_entrada->setMovData(new DateTime());
					$mov_entrada->setMovDataInclusao(new DateTime());
					$mov_entrada->setMovDataInsercao(new DateTime());
					$mov_entrada->setInventario($inventario);
					$mov_entrada->setFornecedor($fornecedor);
					$mov_entrada->setSetorEntrada($setor);
					$mov_entrada->setMovObservacao($this::MOV_OBS);
					$this->salvar($mov_entrada);

					foreach ($itensMovimentoEntrada as $item) {
						$item_mov_entrada = new ItensMovimento();
						$item_mov_entrada->setMovimento($mov_entrada);
						$item_mov_entrada->setProduto($this->_getOneObject(Produto::class, $item['produto']['id']));
						$item_mov_entrada->setIteLote($item['lote']);
						$item_mov_entrada->setIteValidade(new DateTime($item['validade']));
						$item_mov_entrada->setIteStatus(ItensMovimento::FECHADO);
						$item_mov_entrada->setIteQuantidade(abs($item['quantidade'] - $item['quantidadeOriginal']));
						$item_mov_entrada->setIteConsolidado(ItensMovimento::CONSOLIDADO_SIM);
						$item_mov_entrada->setIteDose($item['doseAtual']);
						$item_mov_entrada->setIteDoseLote($item['doseLote']);

						if ($this->salvar($item_mov_entrada)) {
							$this->deletar(InventarioProdutoTmp::class, $item['id']);
						}
					}
				}

				if (count($err_pro_horus) > 0) {
					$retorno['no_catmat'] = $err_pro_horus;
				}

				if (count($err_server_horus) > 0) {
					$retorno['msg_erro_horus'] = $err_server_horus;
				}

				$repo = $this->_getObject(InventarioProdutoTmp::class, [
					"inventario" => $inv_codigo,
					"status" => InventarioProdutoTmp::ALTERADO
				]);
				if (count($repo) == 0) {
					$inventario->setDataDigitacao(new DateTime());
					$inventario->setDataFinalizacao(new DateTime());
					$inventario->setStatus(Inventario::FECHADO);
					$this->salvar($inventario);
					// $query = $this->getEm()->createNativeQuery("TRUNCATE TABLE inventario_produto_tmp RESTART IDENTITY", new ResultSetMapping());

					// if (empty($query->getResult())) {
						$retorno["inventario_msg"] = "Inventário finalizado com sucesso!";
					// } else {
					// 	$retorno =  "Erro ao processar o inventário";
					// }
				}
			}
			else {
				$retorno = "Não há produtos alterados no inventário";
			}

			return $retorno;
		} catch (Exception $e) {
			return $e->getMessage();
		}
	}

	private function getAllProdutoInventario($grupo, $setor) {
		try {
			$sql = "SELECT DISTINCT
										 ps.set_codigo,
										 s.set_nome ,
										 p.pro_codigo,
										 p.pro_nome,
										 g.gru_codigo,
										 g.gru_nome,
										 upper(im.ite_lote) as ite_lote,
										 im.ite_validade,
										 sa.sal_qtde,
										 sa.sal_dose_atual,
										 sa.sal_dose_lote
						  FROM produto_setor ps
									 JOIN produto p ON ps.pro_codigo = p.pro_codigo
									 JOIN grupo g ON p.gru_codigo = g.gru_codigo
									 JOIN setor s ON s.set_codigo = ps.set_codigo
									 JOIN itens_movimento im ON im.pro_codigo = p.pro_codigo
									 			AND im.ite_validade >= now()
									 JOIN saldo sa ON p.pro_codigo = sa.pro_codigo
												AND s.set_codigo = sa.set_codigo
												AND upper(im.ite_lote) = upper(sa.sal_lote)
							WHERE s.set_codigo = :setor
										AND g.gru_codigo = :grupo
							GROUP BY 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11
							ORDER BY p.pro_nome, im.ite_validade DESC";
			$fields = [
				"set_codigo",
				"set_nome",
				"pro_codigo",
				"pro_nome",
				"gru_codigo",
				"gru_nome",
				"ite_lote",
				"ite_validade",
				"sal_qtde",
				"sal_dose_atual",
				"sal_dose_lote"
			];
			$params = [
				"setor" => $setor,
				"grupo" => $grupo
			];

			return $this->queryNative($sql, $fields, $params);
		} catch (Exception $e) {
			die("Erro: " . $e->getMessage());
		}
	}

	private function excluiRegistrosInventario($inv_codigo, $inventarioProdutoTmp, $inventarioLoteTmp = false) {
		if ($inventarioLoteTmp) {
			foreach ($inventarioLoteTmp as $key => $lote) {
				$this->deletar(InventarioLoteTmp::class, $lote['id']);
			}
		}
		foreach ($inventarioProdutoTmp as $key => $invProd) {
			$this->deletar(InventarioProdutoTmp::class, $invProd['id']);
		}
		$this->deletar(Inventario::class, $inv_codigo);

		return true;
	}

}
