<?php

Zend_Loader::loadClass("Elotech_Db_Table_Abstract");

/**
 * Essa agenda é resposável pelo novo agendamento
 * @tod o agendar exames e consultas, por quantidade (cota) e valor
 */
class Application_Model_Agenda extends Elotech_Db_Table_Abstract
{

    protected $_name = 'agenda';
    protected $_primary = 'age_codigo';
    protected $_dependentTables = array();
    const COMPLETO = 'C';
    const MENSAGEM_COTA_DISPONIVEL = "Cota disponível";

    /**
     * Insert ou update na agenda, com seus filhos
     * @param array $data dados do formulário
     * @return int chave primária do registro inserido ou atualizado
     */
    public function salvar(array $data)
    {
        $this->valoresPadrao($data);
        $this->notEmpty(array("usu_codigo"), $data);
        $this->solicitanteInternoOuExterno($data);
        $this->peloMenosUm(array("med_codigo", "usr_codigo_medico"), $data);
        $this->emptyToUnset($data);

        $itens = $data['itens'];
        unset($data['itens']);

        $tbConv = new Application_Model_Convenio();
        $tbConI = new Application_Model_ConvenioItens();
        $conv_codigo = $tbConI->find(key($itens))->current()->conv_codigo;

        $qtdAgendamentoAtual = count($itens);

        $examesAgendados = implode(',', array_keys($itens));
        $valorAgendamentoAtual = $tbConI->getValorAgendamentoAtual($examesAgendados);

        //Busca os dados do convênio, que serão utilizados nas validações.
        $convenioDados = $tbConv->getDados($conv_codigo);

        $alerta = $this->validarLimitesMaximosGeraisConvenio($convenioDados, $valorAgendamentoAtual);
        if (!empty($alerta)) {
            throw new Zend_Validate_Exception($alerta);
        }

        $tbFun = new Application_Model_Funcoes();
        $arrFinal = array();

        foreach ($itens as $coni_codigo => $dia) {
            $arrConi = array();

            $dia = str_replace('/', '-', $dia);
            $dia = date("Y-m-d", strtotime($dia));

            $vagas = $this->vagas($coni_codigo, $dia, $dia);
            $itemDados = $tbConI->getDados($coni_codigo);

            $arrConi[0]['dia'] = $dia;
            $arrConi[0] = $this->validarDiaConvenio($arrConi[0], $convenioDados, $itemDados, $qtdAgendamentoAtual, $valorAgendamentoAtual, $coni_codigo);

            if (!empty($arrConi[0]['mensagem']) && ($arrConi[0]['mensagem'] != self::MENSAGEM_COTA_DISPONIVEL)) {
                throw new Zend_Validate_Exception($arrConi[0]['mensagem']);
            }
        }

        $this->getDefaultAdapter()->beginTransaction();
        try {
            $age_codigo = parent::salvar($data);

            $tbAgeI = new Application_Model_AgendaItens();

            $tbAgeI->salvarDoArray($itens, $age_codigo, $data);

            $this->getDefaultAdapter()->commit();
        } catch (Exception $exc) {
            $this->getDefaultAdapter()->rollBack();
            throw new Zend_Validate_Exception($exc->getMessage());
        }

        return $age_codigo;
    }

    /**
     * Trabalha os dados recebidos e diz se o médico solicitante é interno ou externo
     * @param array $data dados do formulário
     */
    private function solicitanteInternoOuExterno(&$data)
    {
        if (!$data['interno']) { // 1=interno, 0=externo
            $data["med_codigo"] = $data["usr_codigo_medico"];
            unset($data["usr_codigo_medico"]);
        }
        unset($data['interno']);
    }

    public function geraCodigoBarras($age_codigo)
    {
        $db = Zend_Db_Table_Abstract::getDefaultAdapter();
        $sql = "SELECT gerar_codbarras($age_codigo) as codbarras";
        $stmt = $db->query($sql);
        return $stmt->fetchAll()[0]['codbarras'];
    }

    /**
     * Retorna os dados do agendamento
     * @param int $age_codigo
     * @return Zend_Db_Table_Rowset_Abstract
     */
    public function getAgendamento($age_codigo, $coletados = FALSE)
    {
        $tbConf = new Application_Model_Configuracao();
        $valor = $tbConf->getConfig('VALOR_GUIA_EXAME');
        $coni_valor = "";
        if ($valor == 1) {
            $coni_valor = "coni_valor";
        }

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("age" => "agenda"), "usu_codigo")
            ->join(array("agei" => "agenda_itens"), "agei.age_codigo=age.age_codigo", "agei_data")
            ->join(array("coni" => "convenio_itens"), "coni.coni_codigo=agei.coni_codigo", array("$coni_valor"))
            ->join(array("conv" => "convenio"), "conv.conv_codigo=coni.conv_codigo", "")
            ->join(array("med" => "medico"), "med.med_codigo=conv.med_codigo", array("med_nome", "med_endereco", "med_end_telefone", "med_end_celular"))// joinLeft + joinLeft(unidade)
            ->joinLeft(array("medp" => "medico"), "age.med_codigo=medp.med_codigo", array("medico_e" => "med_nome"))
            ->joinLeft(array("usr" => "usuarios"), "age.usr_codigo_medico=usr.usr_codigo", "usr_nome")
            ->joinLeft(array("cid" => "cidade"), "cid.cid_codigo=med.cid_codigo", "cid_nome")
            ->joinLeft(array("uni" => "unidade"), "uni.uni_codigo=age.uni_codigo", "uni_desc")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_nome", "proc_codigo_sus"))
            ->join(array("usu" => "usuario"), "age.usu_codigo=usu.usu_codigo", "usu_nome")
            ->where("age.age_codigo=?", $age_codigo)
            ->order("proc_nome");

        if ($coletados == 1) {
            $where->join(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "col_data_entrega"); // apenas os coletados
        } else {
            $where->joinLeft(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "col_data_entrega");
        }
        return $this->fetchAll($where);
    }

    /**
     * Retorna quais as orientações para o agendamento informado
     * @param int $age_codigo
     * @return Zend_Db_Table_Rowset_Abstract
     */
    public function getOrientacoes($age_codigo)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), "")
            ->join(array("agei" => "agenda_itens"), "agei.age_codigo=age.age_codigo", "")
            ->join(array("coni" => "convenio_itens"), "coni.coni_codigo=agei.coni_codigo", array("coni_codigo", "proc_codigo"))
            ->where("age.age_codigo=$age_codigo");
        $coni_itens = $this->fetchAll($where)->toArray();
        $retorno = array();
        foreach ($coni_itens as $coni_item) {
            $retorno = array_merge_recursive($retorno, $this->buscarOrientacoesPorConvenioItem($coni_item));
        }
        $retorno = array_unique($retorno, SORT_ASC);
        return $retorno;
    }

    public function buscarOrientacoesPorConvenioItem($coni_item)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("coni" => "convenio_itens"), "")
            ->joinLeft(array("cio" => "convenio_itens_orientacao"), "coni.coni_codigo=cio.coni_codigo", "")
            ->join(array("ori" => "orientacoes_exames"), "ori.ori_exa_codigo=cio.ori_exa_codigo", "ori_exa_orientacoes")
            ->where("coni.coni_codigo =?", $coni_item[coni_codigo]);
        $coni_ori = $this->fetchAll($where);
        if (count($coni_ori) > 0) {
            return $coni_ori->toArray();
        } else {
            $where = $this->select(FALSE)
                ->setIntegrityCheck(FALSE)
                ->distinct()
                ->from(array("coni" => "convenio_itens"), "")
                ->join(array("proo" => "procedimento_orientacoes"), "proo.proc_codigo=coni.proc_codigo", "")
                ->join(array("ori" => "orientacoes_exames"), "ori.ori_exa_codigo=proo.ori_exa_codigo", "ori_exa_orientacoes")
                ->where("coni.proc_codigo=?", $coni_item[proc_codigo]);
            return $this->fetchAll($where)->toArray();
        }
    }

    /**
     * Valores padrão do insert/update
     * @param array $data valores do insert
     */
    private function valoresPadrao(&$data)
    {
        if (empty($data['age_data_insert'])) {
            $data['age_data_insert'] = date("Y-m-d H:i:s");
        }

        if (empty($data['usr_codigo'])) {
            $tbUsr = new Application_Model_Usuarios();
            $data['usr_codigo'] = $tbUsr->getUsrAtual()->usr_codigo; // pode gerar exception
        }
    }

    /**
     * Retorna quantas vagas há entre as datas informadas
     * @param array $coni_codigos lista de itens do convenio
     */
    public function getVagas($coni_codigos, $data_inicial, $data_final)
    {
        $coni_codigos = (array)$coni_codigos;

        $tbConv = new Application_Model_Convenio();
        $tbConI = new Application_Model_ConvenioItens();
        $conv_codigo = $tbConI->find($coni_codigos[0])->current()->conv_codigo;

        // verificar se o local(L/H/U) atende aos sábados e domingos
        $sd = $tbConv->atendeSabadoEDomingo($conv_codigo);

        $tbFun = new Application_Model_Funcoes();
        $arrDatas = $tbFun->datasToArray($data_inicial, $data_final);
        $arrFinal = array();

        foreach ($coni_codigos as $coni_codigo) {
            $arrConi = array();
            $vagas = $this->vagas($coni_codigo, $data_inicial, $data_final);
            foreach ($vagas as $dia) {
                if ($dia->limite_dia < 0)
                    $vagasRestantesDia = -1;
                else
                    $vagasRestantesDia = $dia->limite_dia - $dia->agendado_dia;

                if ($dia->limite_mes < 0)
                    $vagasRestantesMes = -1;
                else
                    $vagasRestantesMes = $dia->limite_mes - $dia->agendado_mes;


                if (!$this->sabadoDomingoOuFeriado($dia->grad_dia, $sd))
                    $arrConi[$dia->grad_dia] = 0; // não podem haver vagas nos sábados*, domingos* e feriados
                else
                    $arrConi[$dia->grad_dia] = $this->vagasRestantes($vagasRestantesDia, $vagasRestantesMes);
            }

            if (count($arrConi) < count($arrDatas)) {
                $tbGram = new Application_Model_GradeMes();
                $tbGrad = new Application_Model_GradeDia();

                foreach ($arrDatas as $dia) {
                    if (!array_key_exists($dia, $arrConi)) {
                        if (!$this->sabadoDomingoOuFeriado($dia, $sd)) {
                            $arrConi[$dia] = 0; // não podem haver vagas nos sábados*, domingos* e feriados
                        } else {
                            $limite_dia = $tbGrad->getCotaDia($coni_codigo, $dia);
                            $limite_mes = $tbGram->getCotaMes($coni_codigo, $dia);
                            $arrConi[$dia] = min(array($limite_dia, $limite_mes));
                        }
                    }
                }
            }
            ksort($arrConi);
            $arrFinal [$coni_codigo] = $arrConi;
        }

        return $arrFinal;
    }

    /*     * *
    Nova função para o calendário do agendamento do laboratório.
    Foram adicionadas algumas validações com relação aos convênios.
    As mesmas são disparadas conforme o tipo do convênio, pois os convênios Simples não possuem os valores para validar.
    * * */

    public function getVagasNovo($coni_codigos, $data_inicial, $data_final)
    {
        $coni_codigos = (array)$coni_codigos;
        $tbConv = new Application_Model_Convenio();
        $tbConI = new Application_Model_ConvenioItens();
        $conv_codigo = $tbConI->find($coni_codigos[0])->current()->conv_codigo;

        $qtdAgendamentoAtual = count($coni_codigos);

        $examesAgendados = implode(',', $coni_codigos);
        $valorAgendamentoAtual = $tbConI->getValorAgendamentoAtual($examesAgendados);

        //Busca os dados do convênio, que serão utilizados nas validações.
        $convenioDados = $tbConv->getDados($conv_codigo);

        $alerta = $this->validarLimitesMaximosGeraisConvenio($convenioDados, $valorAgendamentoAtual);

        if (!empty($alerta))
            return $alerta;

        $tbFun = new Application_Model_Funcoes();
        $arrDatas = $tbFun->datasToArray($data_inicial, $data_final);
        $arrFinal = array();

        foreach ($coni_codigos as $coni_codigo) {
            $arrConi = array();
            $itemDados = $tbConI->getDados($coni_codigo);

            foreach ($arrDatas as $dia) {
                if (!array_key_exists($dia, $arrConi)) {
                    $arrConi[$dia]['dia'] = $dia;
                    $arrConi[$dia] = $this->validarDiaConvenio($arrConi[$dia], $convenioDados, $itemDados, $qtdAgendamentoAtual, $valorAgendamentoAtual, $coni_codigo);
                }
            }

            ksort($arrConi);
            $arrFinal [$coni_codigo] = $arrConi;
        }

        return $arrFinal;
    }

    public function validarLimitesMaximosGeraisConvenio($convenioDados, $valorAgendamentoAtual)
    {
        //Quantidade de atendimentos e soma dos valores totais do convênio, dentro do período de vigência
        $tbAgendaItens = new Application_Model_AgendaItens();
        $qtdAtendimentosTotalConvenio = $tbAgendaItens->getQtdAtendimentosTotalConvenio($convenioDados);


        //Convênio completo, quantidade total de atendimentos maior que a permitida, e valor de configuração existente
        if ($convenioDados->tipo_convenio == self::COMPLETO && ($qtdAtendimentosTotalConvenio->quantidade >= $convenioDados->max_total && $convenioDados->max_total != '')) {
            return "Cota máxima do convênio excedida";
        } elseif ($convenioDados->tipo_convenio == self::COMPLETO && (($qtdAtendimentosTotalConvenio->valor + $valorAgendamentoAtual->valor) > $convenioDados->conv_valor_total && $convenioDados->conv_valor_total != '')) {
            return "Valor máximo do convênio excedido";
        }
        return false;
    }

    public function validarDiaConvenio($diaConvenio, $convenioDados, $itemConvenioDados, $qtdAgendamentoAtual, $valorAgendamentoAtual, $coni_codigo)
    {
        $tbAgendaItens = new Application_Model_AgendaItens();
        $tbConv = new Application_Model_Convenio();
        $tbGram = new Application_Model_GradeMes();
        $tbGrad = new Application_Model_GradeDia();

        // verificar se o local(L/H/U) atende aos sábados e domingos
        $sd = $tbConv->atendeSabadoEDomingo($convenioDados->conv_codigo);

        if (!$this->sabadoDomingoOuFeriado($diaConvenio['dia'], $sd)) {
            $diaConvenio['mensagem'] = "Data indisponível (Sábado, Domingo ou Feriado)";
            $diaConvenio['vagas'] = 0;
        } elseif ($convenioDados->tipo_convenio == self::COMPLETO) {
            if ($convenioDados->max_dia != '') {
                $qtdAtendimentosDiaConvenio = $tbAgendaItens->getQtdAtendimentosDiaConvenio($convenioDados->conv_codigo, $diaConvenio['dia']);
            }
            if ($itemConvenioDados->coni_cota_dia != '' && $itemConvenioDados->coni_cota_dia != -1) {
                $qtdAtendimentosDiaConvItem = $tbAgendaItens->getQtdAtendimentosDiaConvItem($coni_codigo, $diaConvenio['dia']);
            }
            if ($convenioDados->dia_mes != '') {
                $qtdAtendimentosMesConvenio = $tbAgendaItens->getQtdAtendimentosMesConvenio($convenioDados, $diaConvenio['dia'])->toArray();
            }
            if (($itemConvenioDados->coni_cota_mes != '' && $itemConvenioDados->coni_cota_mes != -1) && $convenioDados->dia_mes != '') {
                $qtdAtendimentosMesConvItem = $tbAgendaItens->getQtdAtendimentosMesConvItem($coni_codigo, $convenioDados->conv_codigo, $diaConvenio['dia']);
            }

            if (($convenioDados->data_inicial != '' && $convenioDados->data_final != '') && ($diaConvenio['dia'] < $convenioDados->data_inicial || $diaConvenio['dia'] > $convenioDados->data_final)) {
                $diaConvenio['mensagem'] = "Período do convênio excedido";
                $diaConvenio['vagas'] = 0;
            } elseif (($itemConvenioDados->coni_cota_dia != '' && $itemConvenioDados->coni_cota_dia > 0) && $qtdAtendimentosDiaConvItem->quantidade >= $itemConvenioDados->coni_cota_dia) {
                $diaConvenio['mensagem'] = "Cota diária do exame excedida";
                $diaConvenio['vagas'] = 0;
            } elseif ($convenioDados->max_dia != '' && (($qtdAtendimentosDiaConvenio->quantidade + $qtdAgendamentoAtual) >= $convenioDados->max_dia)) {
                $diaConvenio['mensagem'] = "Cota diária do convênio excedida";
                $diaConvenio['vagas'] = 0;
            } elseif (($itemConvenioDados->coni_cota_mes != '' && $itemConvenioDados->coni_cota_mes > 0) && $qtdAtendimentosMesConvItem->quantidade >= $itemConvenioDados->coni_cota_mes) {
                $diaConvenio['mensagem'] = "Cota mensal do exame excedida";
                $diaConvenio['vagas'] = 0;
            } elseif ($convenioDados->max_mes != '' && ($qtdAtendimentosMesConvenio['quantidade'] + $qtdAgendamentoAtual) >= $convenioDados->max_mes) {
                $diaConvenio['mensagem'] = "Cota mensal do convênio excedida";
                $diaConvenio['vagas'] = 0;
            } elseif ($convenioDados->conv_valor_mensal != '' && (($qtdAtendimentosMesConvenio['valor'] + $valorAgendamentoAtual->valor) > $convenioDados->conv_valor_mensal)) {
                $diaConvenio['mensagem'] = "Valor mensal do convênio excedido";
                $diaConvenio['vagas'] = 0;
            } else {
                $limite_dia = $tbGrad->getCotaDia($coni_codigo, $diaConvenio['dia']);
                $limite_mes = $tbGram->getCotaMes($coni_codigo, $diaConvenio['dia']);
                $diaConvenio['mensagem'] = self::MENSAGEM_COTA_DISPONIVEL;
                $diaConvenio['vagas'] = min(array($limite_dia, $limite_mes));
                if ($diaConvenio['vagas'] == -1) {
                    $diaConvenio['vagas'] = 99999;
                }
            }
        } else {
            $limite_dia = $tbGrad->getCotaDia($coni_codigo, $diaConvenio['dia']);
            $limite_mes = $tbGram->getCotaMes($coni_codigo, $diaConvenio['dia']);
            $diaConvenio['mensagem'] = self::MENSAGEM_COTA_DISPONIVEL;
            $diaConvenio['vagas'] = min(array($limite_dia, $limite_mes));
        }

        return $diaConvenio;
    }

    /**
     * Retornar quantas vagas há por dia
     * @param int $coni_codigo
     * @param date $data_inicial
     * @param date $data_final
     * @return array
     */
    public function getTotalVagas($coni_codigo, $data_inicial, $data_final)
    {
        $arrConi = array();
        // Verifica quantos exames foram foram agendados por dia, quantas vagas há no mês, quantos foram agendados no mês,
        // o mês foi alterado? quanto pode por dia da tabela agenda_itens, grade_dia, grade_mes, nada do convenio itens
        $vagas = $this->vagas($coni_codigo, $data_inicial, $data_final);
        // Pega os dias e coloca um array
        foreach ($vagas as $dia) {
            $arrConi[$dia->grad_dia] = (object)$dia->toArray();
        }
        $tbFun = new Application_Model_Funcoes();
        // Monta um array de datas por meio da data inicial e a data final
        $arrDatas = $tbFun->datasToArray($data_inicial, $data_final);
        // Verfica se a quantidade de dias da grade é menor que os dias atuais
        if (count($arrConi) < count($arrDatas)) {
            $tbGram = new Application_Model_GradeMes();
            $tbGrad = new Application_Model_GradeDia();
            // Iniciando variaveis
            $aux = new stdClass();
            $aux->agendado_dia = 0;
            $aux->agendado_mes = 0;
            // Percorrendo array de datas
            foreach ($arrDatas as $dia) {
                // Verifica se dia não existe no array de grades
                if (!array_key_exists($dia, $arrConi)) {
                    $atendeQueDia = $tbFun->diaSemana($dia);
                    // Validação Laboratório ou Profissional do dia
                    $coni_cota_dia = null;
                    if ($tbGrad->getCotaDiaManExcecao($coni_codigo, $dia, $atendeQueDia)->coni_cota_dia != "") {
                        $coni_cota_dia = $tbGrad->getCotaDiaManExcecao($coni_codigo, $dia, $atendeQueDia)->coni_cota_dia;
                        $agendado_dia = $tbGrad->getCotaDiaManExcecao($coni_codigo, $dia, $atendeQueDia)->agendado_dia;
                    }
                    $dados = array(
                        "agendado_dia" => $agendado_dia,
                        "agendado_mes" => "",
                        "grad_dia" => $dia,
                        "coni_cota_mes" => $tbGram->getCotaMes($coni_codigo, $dia),
                        "limite_mes" => $tbGram->getCotaMes($coni_codigo, $dia),
                        "coni_cota_dia" => $coni_cota_dia,
                        "gram_alterada" => "",
                        "limite_dia" => $coni_cota_dia);

                    $arrConi[$dia] = (object)$dados;
                }
            }
            //Ordena um array pelas chaves em ordem inversa
            ksort($arrConi);
        }

        return $arrConi;
    }

    /**
     * Retornar quantas vagas há por dia, por coni_codigo
     * @param array $coni_codigos
     * @param date $data_inicial
     * @param date $data_final
     * @return array
     */
    public function getTotalVagasArr($coni_codigos, $data_inicial, $data_final)
    {
        $out = array();
        // Percorrendo o array de coni codigo, tanto de exame, quanto de profissional
        foreach ($coni_codigos as $coni_codigo)
            $out [$coni_codigo] = $this->getTotalVagas($coni_codigo, $data_inicial, $data_final);
        return $out;
    }

    /**
     * Verifica se ha um limitador no dia ou mes
     * @param int $dia
     * @param int $mes
     */
    private function vagasRestantes($dia, $mes)
    {
        if ($dia < 0) {
            return $mes;
        } elseif ($dia == 0) {
            return 0;
        } else {
            if ($mes < 0)
                return $dia;

            return min(array($dia, $mes));
        }
    }

    /**
     * Retorna quantas vagas foram usadas por dia, quantas há disponíveis por dia e quantas há disponível por mês
     * @see SQL Anderson
     * @see Zend_Db_Expr - http://stackoverflow.com/questions/6663473/zend-db-select-left-join-on-a-subselect
     * @return Zend_Db_Table_Rowset_Abstract
     */
    public function vagas($coni_codigo, $data_inicial = FALSE, $data_final = FALSE)
    {
        $sql1 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("age" => "agendamento"), "count(age_codigo)")// quantos exames foram foram agendados por dia
            ->where("coni_codigo =?", $coni_codigo)
            ->where("age_data=grad_dia");

        $sql2 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("gram" => "grade_mes"), array("gram_cota_mes"))// quantas vagas há no mês
            ->where("to_char(gram_mes,'mm/yyyy') = to_char(grad_dia,'mm/yyyy')")
            ->where("gram.coni_codigo=$coni_codigo");

        $sql3 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("agei" => "agenda_itens"), "count(agei_codigo)")// quantos foram agendados no mês
            ->where("to_char(agei_data,'mm/yyyy') = to_char(grad_dia,'mm/yyyy')")
            ->where("agei.coni_codigo=$coni_codigo");

        $sql4 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("gram" => "grade_mes"), array("gram_alterada"))// o mês foi alterado?
            ->where("to_char(gram_mes,'mm/yyyy') = to_char(grad_dia,'mm/yyyy')")
            ->where("gram.coni_codigo=$coni_codigo");

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)// quanto pode por dia
            ->from(array("grad" => "grade_dia"), array("grad_dia", "limite_dia" => "grad_cota_dia", "grad_alterada"))
            ->columns(array(
                "agendado_dia" => new Zend_Db_Expr('(' . $sql1->assemble() . ')'),
                "limite_mes" => new Zend_Db_Expr('(' . $sql2->assemble() . ')'),
                "agendado_mes" => new Zend_Db_Expr('(' . $sql3->assemble() . ')'),
                "gram_alterada" => new Zend_Db_Expr('(' . $sql4->assemble() . ')')
            ))
            ->join(array("coni" => "convenio_itens"), "coni.coni_codigo=grad.coni_codigo", array("coni_cota_mes", "coni_cota_dia"))
            ->where("coni.coni_codigo=?", $coni_codigo)
            ->order("grad_dia");

        if ($data_inicial)
            $where->where("grad_dia >= '$data_inicial'");

        if ($data_final)
            $where->where("grad_dia <= '$data_final'");

        return $this->fetchAll($where);
    }

    /**
     * Informa a data final, para montar o grid de agendamento
     * Também valida a data inicial, impedindo data passada
     * @param date $data_inicial
     * @param bool $fimDoMes Verdadeiro para retornar o ultimo dia do mês, falso para retornar N* dias (N = configuração)
     * @return date
     */
    public function calculaDataFinal($data_inicial, $fimDoMes = FALSE)
    {
        $tbConf = new Application_Model_Configuracao();
        $dias = $tbConf->getConfig('AGENDA_MOSTRAR_N_OPCOES');
        $dtRetro = $tbConf->getDadosConfigPelaChave('AGENDA_MOSTRAR_N_OPCOES')->conf_valor_int;
        $data = new DateTime($data_inicial);

        if (empty($dtRetro)) {
            $data_inicial = new DateTime();
        }

        $aux = "P" . $dias . "D";
        $data->add(new DateInterval($aux));

        return $data->format("Y-m-d");
    }

    /**
     * Diz se pode haver atendimento nesse dia
     * @param date $data
     * @param stdClass $sd se o local atende no sábado e domingo
     * @see Application_Model_Convenio::atendeSabadoEDomingo
     */
    private function sabadoDomingoOuFeriado($data, $sd)
    {
        list($y, $m, $d) = explode("-", $data);
        $mk = mktime(0, 0, 0, $m, $d, $y);

        // se não pode fazer no sábado:
        if (!$sd->sabado && date("w", $mk) == 6) {
            return FALSE;
        }

        // se não pode fazer no domingo:
        if (!$sd->domingo && date("w", $mk) == 0) {
            return FALSE;
        }

        // se não pode fazer no feriado:
        $tbFer = new Application_Model_Feriado();
        if (!$sd->feriados && $tbFer->ehFeriado($data)) {
            return FALSE;
        }

        return TRUE;
    }

    /**
     * Retorna o agendamentos de exame do paciente
     * Atenção: local do exame: somente med_codigo
     * @param int $usu_codigo
     * @return Zend_Db_Table_Rowset_Abstract
     */
    public function getHistoricoDeExames($usu_codigo, $data_inicial = FALSE, $data_final = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("age" => "agenda"), array("age_codigo", "age_data_insert", "ate_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", array("agei_codigo", "agei_data", "agei_status"))
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array("usr_nome"))
            ->join(array("coni" => "convenio_itens"), "agei.coni_codigo=coni.coni_codigo", array())
            ->join(array("conv" => "convenio"), "conv.conv_codigo = coni.conv_codigo", array("conv_codigo"))
            ->join(array("med" => "medico"), "med.med_codigo=conv.med_codigo", array("med_nome"))
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_nome"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome"))
            ->where("age.usu_codigo=?", $usu_codigo)
            ->where("agei_status = 'R'")
            ->order("agei_data DESC");
        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);
        return $this->fetchAll($where);
    }

    public function getHistoricoDeExamesTela($usu_codigo, $data_inicial = FALSE, $data_final = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("age" => "agenda"), array("age_codigo", "age_data_insert", "ate_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", array("agei_codigo", "agei_data", "agei_status"))
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array("usr_nome"))
            ->join(array("coni" => "convenio_itens"), "agei.coni_codigo=coni.coni_codigo", array())
            ->join(array("conv" => "convenio"), "conv.conv_codigo = coni.conv_codigo", array("conv_codigo"))
            ->join(array("med" => "medico"), "med.med_codigo=conv.med_codigo", array("med_nome"))
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_nome"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome"))
            ->where("age.usu_codigo=?", $usu_codigo)
            ->order("agei_data DESC");
        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);
        return $this->fetchAll($where);
    }

    public function getDadosUsuarioPorAgendamento($age_codigo = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("a" => "agenda"))
            ->join(array("u" => "usuario"), "u.usu_codigo=a.usu_codigo", array("usu_nome", "usu_datanasc", "usu_mae", "usu_sexo", "usu_celular"))
            ->joinLeft(array("dom" => "domicilio"), "dom.dom_codigo=u.dom_codigo", array("dom_numero", "dom_telefone"))
            ->joinLeft("rua", "rua.rua_codigo=dom.rua_codigo", "rua_nome")
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo=a.usr_codigo_medico", "usr_nome")
            ->joinLeft(array("med" => "medico"), "med.med_codigo=a.med_codigo", "med_nome")
            ->joinLeft(array("uni" => "unidade"), "uni.uni_codigo=a.uni_codigo", "uni_desc")
            ->where("a.age_codigo=$age_codigo");
        return $this->fetchRow($where);
    }

    public function atualizarUsu($de, $para)
    {
        $de = (array)$de;

        $data = array("usu_codigo" => $para);
        $where = $this->select()->where("usu_codigo IN (?)", $de)->getPart(Zend_Db_Table_Select::WHERE);
        $where = $where[0];

        Zend_Registry::get("logger")->log("Atualizando usuarios em " . $this->_name, Zend_Log::INFO);

        return $this->update($data, $where);
    }

    public function relPaciente($usu_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE, $solicitante = FALSE, $interno = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array("age_codigo", "age_data_insert", "ate_codigo", "usu_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", "")
            ->join(array("coni" => "convenio_itens"), "coni.coni_codigo=agei.coni_codigo", "")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", "")
            ->join(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "")
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array("usr_nome"))
            ->joinLeft(array("med" => "medico"), "med.med_codigo=age.med_codigo", array("med_nome"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome", "usu_datanasc", "usu_sexo", "usu_prontuario"))
            ->order("usu.usu_nome");

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        if ($usu_codigo)
            $where->where("age.usu_codigo=?", $usu_codigo);

        if ($interno == 1)
            $where->where("age.usr_codigo_medico = $solicitante");
        else if ($interno == 0 && $solicitante != "")
            $where->where("age.med_codigo = $solicitante");
        return $this->fetchAll($where);
    }

    public function relPacientePorPrestador($usu_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE, $solicitante = FALSE, $interno = FALSE, $prestador = FALSE, $status = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array("age_codigo", "age_data_insert", "ate_codigo", "usu_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", "")
            ->join(array("coni" => "convenio_itens"), "coni.coni_codigo=agei.coni_codigo", "")
            ->join(array("conv" => "convenio"), "coni.conv_codigo=conv.conv_codigo", "")
            ->join(array("prestador" => "medico"), "conv.med_codigo=prestador.med_codigo", array("prestador.med_codigo as cod_prestador", "prestador.med_nome as prestador_nome"))
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", "")
            ->joinLeft(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "")
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array("usr_nome"))
            ->joinLeft(array("med" => "medico"), "med.med_codigo=age.med_codigo", array("med_nome"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome", "usu_datanasc", "usu_sexo", "usu_prontuario"))
            ->order("usu.usu_nome");

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        if ($usu_codigo)
            $where->where("age.usu_codigo = ?", $usu_codigo);

        if ($prestador)
            $where->where("conv.med_codigo = ?", $prestador);

        if ($status != 'T')
            $where->where("agei.agei_status = ?", $status);

        if ($interno == 1)
            $where->where("age.usr_codigo_medico = $solicitante");
        else if ($interno == 0 && $solicitante != "")
            $where->where("age.med_codigo = $solicitante");
        return $this->fetchAll($where);
    }

    public function relProfissionalSolicitante($usu_codigo = FALSE, $uni_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE, $solicitante = FALSE, $interno = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array("age_codigo", "age_data_insert", "ate_codigo", "usu_codigo"))
            ->join(array("uni" => "unidade"), "age.uni_codigo=uni.uni_codigo", "uni_desc")
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", "")
            ->join(array("coni" => "convenio_itens"), "coni.coni_codigo=agei.coni_codigo", "")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", "")
            ->join(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "col_data_coleta")
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array("usr_nome"))
            ->joinLeft(array("med" => "medico"), "med.med_codigo=age.med_codigo", array("med_nome"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome", "usu_datanasc", "usu_sexo", "usu_prontuario"))
            ->order(array("uni_desc", "usr.usr_nome", "med.med_nome", "usu.usu_nome"));

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        if ($usu_codigo)
            $where->where("age.usu_codigo=?", $usu_codigo);

        if ($uni_codigo)
            $where->where("age.uni_codigo=?", $uni_codigo);

        if ($interno == 1)
            $where->where("age.usr_codigo_medico = $solicitante");
        else if ($interno == 0 && $solicitante != "")
            $where->where("age.med_codigo = $solicitante");
        return $this->fetchAll($where);
    }

    public function relExamesRealizados($proc_codigo = FALSE, $usu_codigo = FALSE, $uni_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE, $solicitante = FALSE, $interno = FALSE, $tipo_relatorio = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array(""))
            ->join(array("uni" => "unidade"), "age.uni_codigo=uni.uni_codigo", array("uni_desc", "uni_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", "")
            ->joinLeft(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "")
            ->join(array("coni" => "convenio_itens"), "agei.coni_codigo=coni.coni_codigo", "")
            ->join(array("conv" => "convenio"), "conv.conv_codigo=coni.conv_codigo", "")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_codigo", "proc_nome", "proc_codigo_sus", "count(*) as qtde"))
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array(""))
            ->joinLeft(array("med" => "medico"), "med.med_codigo=age.med_codigo", array(""))
            ->order(array("uni.uni_desc ASC", "proc.proc_nome ASC"))
            ->where("agei.agei_status <> 'C'")
            ->group(array("proc_nome", "proc.proc_codigo", "proc_codigo_sus", "uni.uni_desc", "uni.uni_codigo"));

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        if ($usu_codigo)
            $where->where("age.usu_codigo=?", $usu_codigo);

        if ($uni_codigo)
            $where->where("age.uni_codigo=?", $uni_codigo);

        if ($proc_codigo)
            $where->where("proc.proc_codigo=?", $proc_codigo);

        if ($interno == 1)
            $where->where("age.usr_codigo_medico = $solicitante");

        if ($tipo_relatorio == 'R')
            $where->where("agei.agei_status = 'R'");

        else if ($interno == 0 && $solicitante != "")
            $where->where("age.med_codigo = $solicitante");
        return $this->fetchAll($where);
    }

    public function relExamesSolicitados($usu_codigo = FALSE, $uni_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE, $solicitante = FALSE, $interno = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array(""))
            ->join(array("uni" => "unidade"), "age.uni_codigo=uni.uni_codigo", array("uni_desc", "uni_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", array("agei_data", "COUNT(agei.age_codigo) as quantidade"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome", "usu_prontuario", "usu_datanasc", "usu_sexo"))
            ->order(array("uni.uni_desc ASC", "agei_data ASC", "usu_nome ASC"))
            ->where("agei.agei_status = 'R'")
            ->group(array("uni.uni_desc", "uni.uni_codigo", "usu_nome", "agei_data", "usu_prontuario", "usu_datanasc", "usu_sexo"));

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        if ($usu_codigo)
            $where->where("age.usu_codigo=?", $usu_codigo);

        if ($uni_codigo)
            $where->where("age.uni_codigo=?", $uni_codigo);

        if ($interno == 1)
            $where->where("age.usr_codigo_medico = $solicitante");


        else if ($interno == 0 && $solicitante != "")
            $where->where("age.med_codigo = $solicitante");
        return $this->fetchAll($where);
    }

    public function relProcedimento($usu_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE, $proc_codigo = FALSE, $med_codigo = false)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array(""))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", "")
            ->join(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "")
            ->join(array("coni" => "convenio_itens"), "agei.coni_codigo=coni.coni_codigo", "")
            ->join(array("conv" => "convenio"), "conv.conv_codigo=coni.conv_codigo", "")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_codigo", "proc_nome", "proc_codigo_sus", "count(*) as qtde"))
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array(""))
            ->joinLeft(array("med" => "medico"), "med.med_codigo=age.med_codigo", array(""))
            ->order("proc_nome")
            ->group(array("proc.proc_codigo", "proc_nome", "proc_codigo_sus"));

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        if ($usu_codigo)
            $where->where("age.usu_codigo=?", $usu_codigo);

        if ($proc_codigo)
            $where->where("coni.proc_codigo=$proc_codigo");

        if ($med_codigo)
            $where->where("conv.med_codigo=$med_codigo");

        return $this->fetchAll($where);
    }

    public function relLabProc($age_codigo)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("age" => "agenda"), "")
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", "")
            ->join(array("coni" => "convenio_itens"), "agei.coni_codigo=coni.coni_codigo", "")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_nome", "proc_codigo", "proc_codigo_sus",))
            ->order("proc_nome")
            ->where("age.age_codigo=$age_codigo");
        return $this->fetchAll($where);
    }

    public function relProcPac($age_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array("age_codigo", "age_data_insert", "ate_codigo", "usu_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", "")
            ->join(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "col_data_coleta")
            ->join(array("coni" => "convenio_itens"), "agei.coni_codigo=coni.coni_codigo", "")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_codigo", "proc_nome"))
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array("usr_nome"))
            ->joinLeft(array("med" => "medico"), "med.med_codigo=age.med_codigo", array("med_nome"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome", "usu_datanasc", "usu_sexo", "usu_prontuario", "usu_mae"))
            ->where("age.age_codigo in ($age_codigo)")
            ->order("usu_nome");

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        return $this->fetchAll($where);
    }

    public function relProcPacPorPrestador($age_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array("age_codigo", "age_data_insert", "ate_codigo", "usu_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", array("agei_data", "agei_valor", "agei_status"))
            ->joinLeft(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "col_data_coleta")
            ->join(array("coni" => "convenio_itens"), "agei.coni_codigo=coni.coni_codigo", "")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_codigo", "proc_nome"))
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array("usr_nome"))
            ->joinLeft(array("med" => "medico"), "med.med_codigo=age.med_codigo", array("med_nome"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome", "usu_datanasc", "usu_sexo", "usu_prontuario", "usu_mae"))
            ->where("age.age_codigo in ($age_codigo)")
            ->order("usu_nome");

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        return $this->fetchAll($where);
    }

    public function relProcPacPorProcedimento($uni_codigo = FALSE, $proc_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE, $tipo_relatorio = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array("age_codigo", "age_data_insert", "ate_codigo", "usu_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", array("agei_data", "agei_status"))
            ->joinLeft(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "col_data_coleta")
            ->join(array("coni" => "convenio_itens"), "agei.coni_codigo=coni.coni_codigo", "")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_codigo", "proc_nome"))
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array("usr_nome"))
            ->joinLeft(array("med" => "medico"), "med.med_codigo=age.med_codigo", array("med_nome"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome", "usu_datanasc", "usu_sexo", "usu_prontuario", "usu_mae"))
            ->where("proc.proc_codigo = ($proc_codigo)")
            ->where("age.uni_codigo = ($uni_codigo)")
            ->where("agei.agei_status <> 'C'")
            ->order("usu_nome");

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        if ($tipo_relatorio == 'R')
            $where->where("agei.agei_status = 'R'");

        return $this->fetchAll($where);
    }

    public function relProcSolPorUnidade($uni_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("age" => "agenda"), array("age_codigo", "age_data_insert", "ate_codigo", "usu_codigo"))
            ->join(array("agei" => "agenda_itens"), "age.age_codigo=agei.age_codigo", array("agei_data", "agei_status"))
            ->joinLeft(array("col" => "coleta"), "col.agei_codigo=agei.agei_codigo", "col_data_coleta")
            ->join(array("coni" => "convenio_itens"), "agei.coni_codigo=coni.coni_codigo", "")
            ->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_codigo", "proc_nome"))
            ->joinLeft(array("usr" => "usuarios"), "usr.usr_codigo = age.usr_codigo_medico", array("usr_nome"))
            ->joinLeft(array("med" => "medico"), "med.med_codigo=age.med_codigo", array("med_nome"))
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", array("usu_nome", "usu_datanasc", "usu_sexo", "usu_prontuario", "usu_mae"))
            ->where("age.uni_codigo = ($uni_codigo)")
            ->where("agei.agei_status <> 'C'")
            ->order("usu_nome");

        if ($data_inicial)
            $where->where("agei_data >= ?", $data_inicial);

        if ($data_final)
            $where->where("agei_data <= ?", $data_final);

        return $this->fetchAll($where);
    }

    public function relPacientesAgendados($data_inicial, $data_final, $uni_codigo = FALSE)
    {
        $select = "uni.uni_desc,
      usu.usu_nome,
      TO_CHAR(agei.agei_data, 'DD/MM/YYYY') AS data_agendamento,
      dom.dom_telefone,
      COUNT(agei.agei_codigo)  AS qtde";
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("age" => "agenda"), $select)
            ->join(array("agei" => "agenda_itens"), "agei.age_codigo = age.age_codigo", "")
            ->join(array("usu" => "usuario"), "usu.usu_codigo = age.usu_codigo", "")
            ->join(array("uni" => "unidade"), "usu.uni_codigo = uni.uni_codigo", "")
            ->joinLeft(array("dom" => "domicilio"), "usu.dom_codigo = dom.dom_codigo", "")
            ->where("agei.agei_data BETWEEN '$data_inicial' AND '$data_final'");
        if ($uni_codigo) {
            $where->where("uni.uni_codigo = $uni_codigo");
        }
        $where->group(array("uni.uni_desc", "usu.usu_nome", "agei.agei_data", "dom.dom_telefone"))
            ->order(array("uni.uni_desc", "agei.agei_data", "usu.usu_nome"));
        return $this->fetchAll($where);
    }

    public function getAgendamentoFila($fiusu_codigo)
    {
        $sql = $this->select()
            ->from(array("age" => "agenda"))
            ->where("age.fiusu_codigo = $fiusu_codigo");
        return $this->fetchRow($sql);

    }
}
