<?php

Zend_Loader::loadClass("Elotech_Db_Table_Abstract");

class Application_Model_Produto extends Elotech_Db_Table_Abstract
{

    protected $_name = 'produto';
    protected $_primary = 'pro_codigo';
    protected $_dependentTables = array('LeitoGradeItens');
    protected $_sequence = 'seq_pro_codigo';

    /**
     * Motivos da perda (vacina)
     */
    const QUEBRA_DE_FRASCO = 1;
    const FALTA_DE_ENERGIA = 2;
    const FALHA_NO_EQUIPAMENTO = 3;
    const VALIDADE_VENCIDA = 4;
    const PROCEDIMENTO_INADEQUADO = 5;
    const FALHA_NO_TRANSPORTE = 6;
    const OUTROS_MOTIVOS = 7;


  public function salvarProduto(array $data) {
    try {
      $this->emptyToNull($data);
      return parent::salvar($data);
    } catch (Exception $exc) {
      throw new Zend_Validate_Exception("Falha ao cadastrar Produto: " . $exc->getMessage());
    }
  }

    /**
     * Busca medicamentos n?o-controlados
     * usado para alimentar o plugin de busca (jquery)
     * @return json
     */
    public function buscarMedicamentosPosto($term = FALSE)
    {
        return $this->buscarMedicamentos($term, FALSE);
    }

    /**
     * Busca medicamentos controlados
     * usado para alimentar o plugin de busca (jquery)
     * @return json
     */
    public function buscarMedicamentosControlado($term = FALSE)
    {
        return $this->buscarMedicamentos($term, 2);
    }

    public function buscarMedicamentos($term = FALSE, $controlado = NULL, $limite = FALSE, $movimento = FALSE)
    {

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"))
            ->joinLeft(array("umed" => "unidmedida"), "pro.umed_codigo = umed.umed_codigo", array("umed_nome"))
            ->where("pro_situacao='A'")
            ->where("permite_prescricao='S'")
            ->order("pro_nome");


        if ($term)
            $where->where("pro_nome ilike '%$term%'");

        switch ($controlado) {
            case 2:
                $where->where("psico_codigo IS NOT NULL");
                break;
            case 1:
                break;
            default:
                $where->where("psico_codigo IS NULL");
                break;
        }

        if ($limite)
            $where->limit($limite);

        if ($movimento)
            $where->join(array("im" => "itens_movimento"), "im.pro_codigo=p.pro_codigo");

        $all = $this->fetchAll($where);

        $out = array();
        foreach ($all as $pro) {
            $out [] = array(
                "id" => $pro->pro_codigo,
                "label" => $pro->pro_nome,
                "data" => array("pro_codigo" => $pro->pro_codigo, "pro_nome" => $pro->pro_nome, "umed_nome" => $pro->umed_nome)
            );
        }

        if (!count($out)) {
            $out [] = array(
                "id" => 0,
                "label" => "Nenhum item encontrado",
                "data" => array()
            );
        }
        return $out;
    }

    public function buscarMedicamentosComMovimentacoes()
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"), array("pro_codigo", "pro_nome", "pro_horus", "pro_horus_tipo"))
            ->distinct("DISTINCT")
            ->join(array("im" => "itens_movimento"), "im.pro_codigo=pro.pro_codigo", "")
            ->where("pro_tipo='M'")
            ->where("pro_situacao='A'")
            ->where("pro_horus is null")
            ->order("pro_nome");
        //die($where);
        return $this->fetchAll($where);
    }

    public function buscarProdutos($term = FALSE, $limit = FALSE, $set_codigo = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"))
            ->where("pro_situacao='A'")
            ->order("pro_nome");

        if ($term)
            $where->where("pro_nome ilike '%$term%'");

        if ($limit)
            $where->limit($limit);
        //**Essa parte estabelece que o produto buscado tem que estar no setor passado no parametro**/

        $all = $this->fetchAll($where);
        $out = array();
        $tbProSet = new Application_Model_ProdutoSetor();
        foreach ($all as $pro) {
            /*             * Essa parte  apenas verifica se o produto tem vinculo com o setor passado no parametro e insere true ou false pra variavel produto_vinculo_setor* */
            if ($set_codigo) {
                $verifica_vinculo_setor = $tbProSet->verificaVinculoProdutoSetor($pro->pro_codigo, $set_codigo);
            } else {
                $verifica_vinculo_setor = 0;
            }
            $out [] = array(
                "id" => $pro->pro_codigo,
                "label" => $pro->pro_nome,
                "data" => array(
                    "pro_codigo" => $pro->pro_codigo,
                    "pro_horus" => $pro->pro_horus,
                    "pro_horus_tipo" => $pro->pro_horus_tipo,
                    "pro_nome" => $pro->pro_nome,
                    "pro_validade" => $pro->pro_validade,
                    "pro_fracionado" => $pro->pro_fracionado,
                    "produto_vinculo_setor" => $verifica_vinculo_setor
                )
            );
        }

        if (!count($out)) {
            $out [] = array(
                "id" => 0,
                "label" => "Nenhum item encontrado",
                "data" => array()
            );
        }
        return $out;
    }

    public function buscarSaldoProdutosComEstoqueSetorSolicitante($pro_codigo, $set_codigo, $tipo = FALSE)
    {
        $where = $this->select()
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"), array())
            ->joinLeft(array("psico" => "psicotropicos"), "psico.psico_codigo=pro.psico_codigo", array())
            ->join(array("proset" => "produto_setor"), "proset.pro_codigo=pro.pro_codigo", array())
            ->join(
                array("sal" => "saldo"),
                "(sal.pro_codigo = pro.pro_codigo AND sal.set_codigo = proset.set_codigo)",
                "SUM(sal.sal_qtde) AS quantidade"
            )
            ->where("proset.set_codigo=$set_codigo")
            ->where("pro.pro_situacao='A'")
            ->where("pro.pro_codigo = $pro_codigo")
            ->group(array("pro.pro_codigo", "proset.prset_minimo", "psico.psico_exige_codigo", "psico.psico_codigo", "proset.prset_codigo"))
            ->order("pro_nome");

//        die($where);
        return $this->fetchRow($where)->quantidade;
    }

    public function buscarProdutosComEstoque($term = FALSE, $limite = FALSE, $set_codigo = FALSE, $tipo = FALSE)
    {

        if ($tipo == "S") {// sem validade
            $tmp = " AND sal.set_codigo = $set_codigo and sal_qtde > 0";
        } else {
            $tmp = " AND sal.set_codigo = $set_codigo and sal_qtde > 0 and sal_validade > CURRENT_DATE";
        }

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"))
            ->joinLeft(array("psico" => "psicotropicos"), "psico.psico_codigo=pro.psico_codigo", "psico_exige_codigo")
            ->join(array("proset" => "produto_setor"), "proset.pro_codigo=pro.pro_codigo", "ROUND(COALESCE(proset.prset_minimo, 0),0) as pro_estoque_minimo")
            ->join(array("sal" => "saldo"), "sal.pro_codigo = pro.pro_codigo AND sal.set_codigo = proset.set_codigo $tmp", "SUM(sal.sal_qtde) AS quantidade")
            ->join(array("ite" => "itens_movimento"), "ite.pro_codigo = pro.pro_codigo", "TRIM(ite_codbarras) AS ite_codbarras")
            ->where("proset.set_codigo=$set_codigo")
            ->where("pro_situacao='A'")
            ->group(array("pro.pro_codigo", "proset.prset_minimo", "psico.psico_exige_codigo", "TRIM(ite.ite_codbarras)"))
            ->order("pro_nome");

        if ($term)
            $where->where("pro_nome||ite_codbarras ilike '%$term%'");

        if ($limite)
            $where->limit($limite);

//        die($where);

        $all = $this->fetchAll($where);
        $out = array();
        foreach ($all as $pro) {
            $out [] = array(
                "id" => $pro->pro_codigo,
                "label" => $pro->pro_nome,
                "count" => $all->count(),
                "data" => array(
                    "pro_codigo" => $pro->pro_codigo,
                    "pro_horus" => $pro->pro_horus,
                    "pro_horus_tipo" => $pro->pro_horus_tipo,
                    "psico_exige_codigo" => $pro->psico_exige_codigo,
                    "pro_nome" => $pro->pro_nome,
                    "pro_validade" => $pro->pro_validade,
                    "pro_fracionado" => $pro->pro_fracionado,
                    "psico_codigo" => $pro->psico_codigo,
                    "pro_estoque_minimo" => $pro->pro_estoque_minimo,
                    "quantidade" => $pro->quantidade,
                    "cod_barras" => $pro->ite_codbarras
                )
            );
        }

        if (!count($out)) {
            $out [] = array(
                "id" => 0,
                "label" => "Nenhum item encontrado",
                "data" => array(),
                "count" => 0
            );
        }
        return $out;
    }

    public function buscarMedicamentosHorus($term = FALSE, $controlado = NULL, $limite = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"))
            ->where("pro_tipo='M'")
            ->where("pro_situacao='A'")
            ->where("gru_codigo=99482")
            ->where("pro_horus is not null")
            ->where("pro_horus_tipo is not null")
            ->where("pro_codigo not in (select distinct pro_codigo from itens_movimento)")
            ->order("pro_nome");

        if ($term)
            $where->where("pro_nome ilike '%$term%'");

        if ($limite)
            $where->limit($limite);

        $all = $this->fetchAll($where);

        $out = array();
        foreach ($all as $pro) {
            $out [] = array(
                "id" => $pro->pro_codigo,
                "label" => $pro->pro_nome,
                "data" => array(
                    "pro_codigo" => $pro->pro_codigo,
                    "pro_nome" => $pro->pro_nome,
                    "horus" => $pro->pro_horus,
                    "horus_tipo" => $pro->pro_horus_tipo
                )
            );
        }

        if (!count($out)) {
            $out [] = array(
                "id" => 0,
                "label" => "Nenhum item encontrado",
                "data" => array()
            );
        }
        return $out;
    }

    /**
     * Seleciona os lotes necess?rios para atingir a quantidade selecionada.
     */
    public function selecionaMelhorLote($produtos, $set_codigo)
    {
        $tbCont = new Application_Model_ControleFracionado();
        $tbSal = new Application_Model_Saldo();
        $saida = array();

        foreach ($produtos as $pro_codigo => $quantidade) {
            $dosesSelecionadas = 0;
            $desse = 0;

            // pega todos os lotes fracionados
            $fracoes = $tbCont->getLotesFracionados($pro_codigo, $set_codigo);
            foreach ($fracoes as $fracao) {
                if ($quantidade == $dosesSelecionadas)
                    break; // sai do foreach;

                if ($quantidade >= $dosesSelecionadas + $fracao->cont_dose) {
                    $pegar = $fracao->cont_dose;
                } else {
                    $pegar = $quantidade - $dosesSelecionadas;
                }

                $dosesSelecionadas += $pegar;
                $desse += $pegar;

                if (!isset($saida[$pro_codigo]))
                    $saida[$pro_codigo] = array(
                        "pro_nome" => $fracao->pro_nome,
                        "solicitado" => $quantidade,
                        "cont" => array()
                    );

                $saida[$pro_codigo]['cont'][$fracao->cont_codigo] = array(
                    "pro_lote" => $fracao->ite_lote,
                    "pro_validade" => $fracao->ite_validade,
                    "total" => $desse
                );
            }

            // h? doses suficientes?
            if ($quantidade > $dosesSelecionadas) {
                $naoFrac = $tbSal->getLotes($pro_codigo, $set_codigo, TRUE, 1);

                // para cada lote/validade
                foreach ($naoFrac as $lote) {
                    $desse = 0;
                    for ($x = 1; $x <= $lote->sal_qtde; $x++) {
                        if ($quantidade == $dosesSelecionadas)
                            break;

                        if ($quantidade >= $dosesSelecionadas + $lote->sal_dose_lote) {
                            $pegar = $lote->sal_dose_lote;
                        } else {
                            $pegar = $quantidade - $dosesSelecionadas;
                        }
                        $dosesSelecionadas += $pegar;
                        $desse += $pegar;

                        if (!isset($saida[$pro_codigo]))
                            $saida[$pro_codigo] = array(
                                "pro_nome" => $lote->pro_nome,
                                "solicitado" => $quantidade
                            );

                        $saida[$pro_codigo]['saldo'][$lote->sal_codigo] = array(
                            "pro_lote" => $lote->sal_lote,
                            "pro_validade" => $lote->sal_validade,
                            "total" => $desse
                        );
                    }
                }
            }

            // h? doses suficientes?
            if ($quantidade > $dosesSelecionadas) {
                if ($dosesSelecionadas == 0) {
                    $saida[$pro_codigo]['pro_nome'] = $this->fetchRow("pro_codigo=$pro_codigo")->pro_nome;
                }
                $saida[$pro_codigo]['solicitado'] = $quantidade;
                $saida[$pro_codigo]['faltou'] = $quantidade - $dosesSelecionadas;
            }
        }

        return $saida;
    }

    /**
     * Seleciona qual o melhor lote/validade do produto em estoque (saldo)
     * Obs.: não olha os produtos j? fracionados
     * @param array $pro_codigo
     */
    public function selecionaMelhorLoteNaoFracionado($pro_codigo, $set_codigo)
    {
        $pro_codigo = (array)$pro_codigo;

        $subSelect3 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("sal_3" => "saldo"), "MIN(sal_validade)")
            ->where("sal_3.pro_codigo=sal_2.pro_codigo")
            ->where("sal_3.set_codigo=sal_2.set_codigo")
            ->where("sal_3.sal_qtde > 0")
            ->where("sal_3.sal_validade >= CURRENT_DATE");


        $subSelect2 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("sal_2" => "saldo"), "MIN(sal_2.sal_qtde)")
            ->where("sal_2.pro_codigo=sal_1.pro_codigo")
            ->where("sal_2.set_codigo=sal_1.set_codigo")
            ->where("sal_2.sal_qtde > 0")
            ->where("sal_2.sal_validade >= CURRENT_DATE")
            ->where("sal_2.sal_validade = ?", $subSelect3);

        $subSelect1 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("sal_1" => "saldo"), "MIN(sal_1.sal_codigo)")
            ->where("sal_1.pro_codigo=sal.pro_codigo")
            ->where("sal_1.set_codigo=sal.set_codigo")
            ->where("sal_1.sal_qtde > 0")
            ->where("sal_1.sal_validade >= CURRENT_DATE")
            ->where("sal_1.sal_qtde = ?", $subSelect2);

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("sal" => "saldo"), array("(pro_codigo)", "sal_codigo", "sal_qtde", "sal_lote", "sal_validade", "sal_dose_lote"))
            ->where("sal.set_codigo=?", $set_codigo)
            ->where("sal.sal_qtde > 0")
            ->where("sal.sal_validade >= CURRENT_DATE")
            ->where("pro_codigo in (?)", $pro_codigo)
            ->where("sal.sal_codigo = ?", $subSelect1)
            ->group(array("pro_codigo", "sal_codigo", "sal_qtde", "sal_lote", "sal_validade", "sal_dose_lote"))
            ->order(array("sal_validade", "sal_qtde"));

        return $this->fetchAll($where);
    }

    /**
     * Soma os totais do produto no setor, que ainda não venceu
     * @param int /array $pro_codigo
     * @param int $set_codigo
     */
    public function totalValido($pro_codigo, $set_codigo)
    {
        $pro_codigo = (array)$pro_codigo;

        $sal_qtde = "COALESCE(sal.sal_qtde,0)
                                 -
                                (SELECT COALESCE(sum(remil_quantidade),0)
                                        FROM requisicao_materiais_itens remi
                                        JOIN requisicao_materiais_itens_lote remil
                                          ON remil.remi_codigo = remi.remi_codigo
                                        JOIN requisicao_materiais rem
                                          ON rem.rem_codigo=remi.rem_codigo
                                   WHERE remi.remi_status = 'E'
                                     AND remi.pro_codigo = sal.pro_codigo
                                     AND set_codigo_sol = $set_codigo
                                     AND remil.remil_lote = sal.sal_lote) AS sal_qtde";

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("sal" => "saldo"), array("pro_codigo", "$sal_qtde", "sal_qtde as saldo_original"))
            ->where("pro_codigo in (?)", $pro_codigo)
            ->where("set_codigo=?", $set_codigo)
            ->where("sal_qtde > 0")
            ->where("sal_validade > CURRENT_DATE")
            ->group("pro_codigo")
            ->group("sal_qtde")
            ->group("sal_lote");
        return $this->fetchAll($where);
    }

    /**
     * Busca qual o melhor lote (j? fracionado) de cada produto do array
     * @param array $pro_codigo
     * @param int $set_codigo
     * @return Zend_Db_Table_Rowset_Abstract
     */
    public function selecionaMelhorLoteFracionado($pro_codigo, $set_codigo)
    {
        $pro_codigo = (array)$pro_codigo;

        $subSelect3 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("cont_4" => "controlefracionado"), "")
            ->join(array("ite_4" => "itens_movimento"), "ite_4.ite_codigo=cont_4.ite_codigo", "MIN(ite_4.ite_validade)")
            ->where("cont_4.cont_dose > 0")
            ->where("cont_4.set_codigo=cont_3.set_codigo")
            ->where("ite_4.pro_codigo=ite_3.pro_codigo");

        $subSelect2 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("cont_3" => "controlefracionado"), "MIN(cont_dose)")
            ->join(array("ite_3" => "itens_movimento"), "ite_3.ite_codigo=cont_3.ite_codigo", "")
            ->where("cont_3.cont_dose > 0")
            ->where("cont_3.set_codigo=cont_2.set_codigo")
            ->where("ite_3.pro_codigo=ite_2.pro_codigo")
            ->where("ite_3.ite_validade in (?)", $subSelect3);

        $subSelect1 = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("cont_2" => "controlefracionado"), "MIN(cont_codigo)")
            ->join(array("ite_2" => "itens_movimento"), "ite_2.ite_codigo=cont_2.ite_codigo", "")
            ->where("cont_2.cont_dose > 0")
            ->where("cont_2.set_codigo=cont.set_codigo")
            ->where("ite_2.pro_codigo=ite.pro_codigo")
            ->where("cont_2.cont_dose in (?)", $subSelect2);


        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("cont" => "controlefracionado"), array("cont_codigo", "cont_dose", "ite_codigo"))
            ->join(array("ite" => "itens_movimento"), "ite.ite_codigo=cont.ite_codigo", array("pro_codigo", "ite_lote", "ite_validade", "fab_codigo"))
            ->where("cont_dose > 0")
            ->where("set_codigo = ?", $set_codigo)
            ->where("pro_codigo in (?)", $pro_codigo)
            ->where("cont_codigo in (?)", $subSelect1);
        //die($where);

        return $this->fetchAll($where);
    }

    public function selecionaMelhoresVacinas()
    {
        $tbUsr = new Application_Model_Usuarios();
        $set_codigo = $tbUsr->getUsrAtual()->set_codigo;

        $tbCar = new Application_Model_Carteirinha();
        $pro_codigo = $tbCar->getProdutosDaCarteirinha();

        return array(
            "fechado" => $this->totalValido($pro_codigo, $set_codigo)->toArray(),
            "aberto" => $this->selecionaMelhorLoteFracionado($pro_codigo, $set_codigo)->toArray()
        );
    }

    /**
     * Faz um movimento de saida (baixando o saldo) e adiciona na tabela controlefracionado
     * @param int $set_codigo
     * @param int $sal_codigo
     * @param int $quantidade
     * @param int $fracoes se informado, $quantidade ser? a quantidade necess?ria para ter $fracaoes fra??es
     * @return int cont_codigo
     */
    public function fracionarFromSaldo($set_codigo, $sal_codigo, $quantidade = 1, $fracoes = FALSE)
    {
        $tbMov = new Application_Model_Movimento();
        $tbIte = new Application_Model_MovimentoItens();
        $tbCont = new Application_Model_ControleFracionado();
        $tbSaldo = new Application_Model_Saldo();

        Zend_Registry::get("logger")->log("iniciar Trasaction", Zend_Log::INFO);
        $this->getAdapter()->beginTransaction();

        try {
            $saldo = $tbSaldo->fetchRow("sal_codigo=$sal_codigo");
            if ($fracoes)
                $quantidade = ceil($fracoes / (int)$saldo->sal_dose_lote);

            Zend_Registry::get("logger")->log("selecionado melhor lote", Zend_Log::INFO);

            // Faz o movimento
            $dadosMovimento = array(
                "mov_tipo" => Application_Model_Movimento::SAIDA,
                "mov_saida" => Application_Model_Movimento::DISPENSACAO,
                "set_saida" => $set_codigo
            );
            $mov_codigo = $tbMov->salvar($dadosMovimento);
            Zend_Registry::get("logger")->log("adicionado movimento: $mov_codigo", Zend_Log::INFO);

            // Faz o item do movimento
            $dadosItensMovimento = array(
                "mov_codigo" => $mov_codigo,
                "pro_codigo" => $saldo->pro_codigo,
                "ite_lote" => $saldo->sal_lote,
                "ite_validade" => $saldo->sal_validade,
                "ite_dose" => (int)$saldo->sal_dose_lote,
                "ite_quantidade" => $quantidade // s? pode fracionar um por vez
            );
            Zend_Registry::get("logger")->log($dadosItensMovimento, Zend_Log::INFO);
            $ite_codigo = $tbIte->salvar($dadosItensMovimento);
            Zend_Registry::get("logger")->log("adicionado itens_movimento: $ite_codigo", Zend_Log::INFO);

            // salva no controlefracionado
            $dadosControleFracionado = array(
                "cont_dose" => ($saldo->sal_dose_lote * $quantidade),
                "set_codigo" => $set_codigo,
                "ite_codigo" => $ite_codigo
            );

            $cont_codigo = $tbCont->salvar($dadosControleFracionado);

            Zend_Registry::get("logger")->log("adicionado controlefracionado", Zend_Log::INFO);
            $this->getAdapter()->commit();

            return $cont_codigo;
        } catch (Exception $e) {
            Zend_Registry::get("logger")->log("Exception: " . $e->getMessage(), Zend_Log::WARN);
            Zend_Registry::get("logger")->log($e->getTrace(), Zend_Log::INFO);
            echo "set: $set_codigo, sal: $sal_codigo, qtde: $quantidade, fracao: $fracoes\n";
            echo "exc: " . $e->getMessage();
            echo "\n: " . $e->getTraceAsString();
            $this->getAdapter()->rollBack();
        }
    }

    /**
     * Fraciona um produto.
     * @example vacinas => doses
     * @example Febre Amarela (1 frasco) => 25x doses de F.A.
     * @param int $pro_codigo
     * @param int $set_codigo (Opcional)
     */
    public function fracionar($pro_codigo, $set_codigo = FALSE, $sal_codigo = FALSE)
    {
        Zend_Registry::get("logger")->log("Fracionar: $pro_codigo", Zend_Log::INFO);

        if (!$set_codigo) {
            $tbUsr = new Application_Model_Usuarios();
            $set_codigo = $tbUsr->getUsrAtual()->set_codigo;
        }

        // verifica se j? existe fracionado
        $fracionado = $this->selecionaMelhorLoteFracionado($pro_codigo, $set_codigo);
        if ($fracionado->count()) {
            Zend_Registry::get("logger")->log($fracionado->toArray(), Zend_Log::INFO);

            return false;
        }

        $saldo = $this->selecionaMelhorLoteNaoFracionado($pro_codigo, $set_codigo);
        return $this->fracionarFromSaldo($set_codigo, $saldo[0]['sal_codigo']);
    }

    public function fracionarVarios($produtos, $set_codigo = FALSE)
    {

        if (!$set_codigo) {
            $tbUsr = new Application_Model_Usuarios();
            $set_codigo = $tbUsr->getUsrAtual()->set_codigo;

        }

        $cont = array();
        foreach ($produtos as $sal_codigo => $quantidade) {
            $cont_codigo = $this->fracionarFromSaldo($set_codigo, $sal_codigo, false, $quantidade);
            $cont [$cont_codigo] = $quantidade;
        }

        return $cont;
    }

    /**
     * Descarta um produto da tabela controlefracionado
     * @param int $pro_codigo
     * @param int $set_codigo
     */
    public function descartar($pro_codigo, $motivo = self::OUTROS_MOTIVOS, $set_codigo = FALSE)
    {
        if (!$set_codigo) {
            $tbUsr = new Application_Model_Usuarios();
            $set_codigo = $tbUsr->getUsrAtual()->set_codigo;
        }

        // verifica se existe fracionado
        $fracionado = $this->selecionaMelhorLoteFracionado($pro_codigo, $set_codigo);
        if (!$fracionado->count()) { // se não houver
            Zend_Registry::get("logger")->log("Produto $pro_codigo não possui fra??es/doses abertas", Zend_Log::INFO);
            return false;
        }

        $fracionado = $fracionado->current();
        $dados = array(
            "cont_codigo" => $fracionado->cont_codigo,
            "cont_dose" => 0,
            "cont_perda" => $fracionado->cont_dose,
            "cont_perda_motivo" => $motivo,
            "set_codigo" => $set_codigo,
            "ite_codigo" => $fracionado->ite_codigo
        );
        Zend_Registry::get("logger")->log($dados, Zend_Log::INFO);
        $tbCont = new Application_Model_ControleFracionado();
        $tbCont->salvar($dados);
    }

    /* RELATORIOS */

    public function relMedicamentosPorValidade(&$dados, $set_codigo, $data_inicial = FALSE, $data_final = FALSE)
    {
        $dados->title = "Produtos a vencer";
        $dados->params = serialize($_POST);
        $dados->config = array(
            "th" => array("pro_nome" => "Medicamento", "sal_lote" => "Lote", "sal_validade" => "Validade", "sal_qtde" => "Quant."),
            "formato" => array("sal_validade" => "data", "sal_qtde" => "num")
        );

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("sal" => "saldo"), array("sal_lote", "sal_validade", "sal_qtde"))
            ->join(array("pro" => "produto"), "pro.pro_codigo=sal.pro_codigo", array("pro_nome", "pro_horus", "pro_horus_tipo"))
            ->where("sal.sal_qtde > 0")
            ->where("sal.set_codigo=?", $set_codigo)
            ->order("pro_nome");

        if ($data_inicial)
            $where->where("sal.sal_validade >= ?", $data_inicial);

        if ($data_final)
            $where->where("sal.sal_validade <= ?", $data_final);

        return $where;
    }

    public function relMedicamentosPorEstoqueMinimo(&$dados, $set_codigo, $pro_codigo, $pros_codigo, $gru_codigo, $tipo_relatorio)
    {
        $dados->title = "Estoque mínimo - Data de emissão: " . date('d/m/Y');
        $dados->params = serialize($_POST);
        $dados->config = array(
            "th" => array("pro_codigo" => "<div style='text-align:left;'>Código</div>","pro_nome" => "<div style='text-align:left;'>Produto</div>", "prset_minimo" => "<div style='text-align:right;'>Estoque Mínimo</div>", "sal_qtde" => "<div style='text-align:right;'>Saldo</div>"),
            "formato" => array("prset_minimo" => "num", "sal_qtde" => "num")
        );
        if ($tipo_relatorio == "M") {
            $where = $this->select(FALSE)
                ->setIntegrityCheck(FALSE)
                ->from(array("sal" => "saldo"), array("SUM(sal_qtde) as sal_qtde"))
                ->join(array("pro" => "produto"), "pro.pro_codigo=sal.pro_codigo", array("pro_codigo","pro_nome",  "pro_horus", "pro_horus_tipo"))
                ->join(array("proset" => "produto_setor"), "pro.pro_codigo=proset.pro_codigo", "prset_minimo")
                ->joinLeft(array("pros" => "produto_subgrupo"), "pro.pros_codigo=pros.pros_codigo", "pros_codigo")
                ->joinLeft(array("gru" => "grupo"), "pro.gru_codigo=gru.gru_codigo", "gru_codigo")
                ->where("sal.set_codigo=?", $set_codigo)
                ->where("proset.set_codigo=?", $set_codigo)
                ->where("proset.prset_minimo IS NOT NULL")
                ->where("(select sum(sal1.sal_qtde) "
                    . "from saldo sal1 "
                    . "join produto pro1 on pro1.pro_codigo=sal1.pro_codigo "
                    . "join produto_setor proset1 on pro1.pro_codigo=proset1.pro_codigo "
                    . "where sal1.set_codigo=$set_codigo and "
                    . "proset1.set_codigo=$set_codigo and "
                    . "proset1.prset_minimo IS NOT NULL and "
                    . "pro1.pro_codigo = pro.pro_codigo) <= proset.prset_minimo ")
                ->where("pro.pro_situacao = 'A'")
                ->group(array("pro_nome", "prset_minimo", "pro.pro_codigo", "pros.pros_codigo", "gru.gru_codigo"))
                ->order("pro_nome");
            if ($pro_codigo) {
                $where->where("pro.pro_codigo=?", $pro_codigo);
            }
            if ($pros_codigo) {
                $where->where("pro.pros_codigo=?", $pros_codigo);
            }
            if ($gru_codigo) {
                $where->where("pro.gru_codigo=?", $gru_codigo);
            }
        } else {
            $where = $this->select(FALSE)
                ->setIntegrityCheck(FALSE)
                ->from(array("sal" => "saldo"), array("SUM(sal_qtde) as sal_qtde"))
                ->join(array("pro" => "produto"), "pro.pro_codigo=sal.pro_codigo", array("pro_codigo","pro_nome", "pro_horus", "pro_horus_tipo"))
                ->join(array("proset" => "produto_setor"), "pro.pro_codigo=proset.pro_codigo", "prset_minimo")
                ->joinLeft(array("pros" => "produto_subgrupo"), "pro.pros_codigo=pros.pros_codigo", "pros_codigo")
                ->joinLeft(array("gru" => "grupo"), "pro.gru_codigo=gru.gru_codigo", "gru_codigo")
                ->where("sal.set_codigo=?", $set_codigo)
                ->where("proset.set_codigo=?", $set_codigo)
                ->where("proset.prset_minimo IS NOT NULL")
                ->group(array("pro_nome", "prset_minimo", "pro.pro_codigo", "pros.pros_codigo", "gru.gru_codigo"))
                ->order("pro_nome");
            if ($pro_codigo) {
                $where->where("pro.pro_codigo=?", $pro_codigo);
            }
            if ($pros_codigo) {
                $where->where("pro.pros_codigo=?", $pros_codigo);
            }
            if ($gru_codigo) {
                $where->where("pro.gru_codigo=?", $gru_codigo);
            }
        }
        return $where;
    }

    public function relEntradaPsicotropicos(&$dados, $set_codigo, $data_inicial = FALSE, $data_final = FALSE)
    {
        // die("CHEGO");
        $dados->title = "Entrada Psicotrópicos";
        $dados->params = serialize($_POST);
        $dados->config = array(
            "th" => array("mov_nr_nota" => "Número da Nota", "pro_nome" => "Medicamento", "for_nome" => "Fornecedor", "for_cnpj" => "CNPJ", "ite_quantidade" => "Quant. Adquirida"),
            "formato" => array("ite_quantidade" => "num")
        );

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"), array("pro_nome", "pro_horus", "pro_horus_tipo"))
            ->join(array("ite" => "itens_movimento"), "ite.pro_codigo=pro.pro_codigo", "SUM(ite_quantidade) AS ite_quantidade")
            ->join(array("mov" => "movimento"), "mov.mov_codigo=ite.mov_codigo", array("mov_nr_nota"))
            ->join(array("f" => "fornecedor"), "f.for_codigo=mov.for_codigo", array("for_nome", "for_cnpj"))
            ->join(array("set" => "setor"), "set.set_codigo=mov.set_entrada")
            ->where("pro.psico_codigo is not null")
            ->where("mov.mov_tipo = 'E'")
            ->where("set.set_codigo=?", $set_codigo)
            ->group(array("for_nome", "pro_nome", "for_cnpj", "set.set_codigo", "mov_nr_nota", "pro_horus", "pro_horus_tipo"))
            ->order("pro_nome");

        if ($data_inicial)
            $where->where("mov.mov_data>= ?", $data_inicial);

        if ($data_final)
            $where->where("mov.mov_data <= ?", $data_final);

        return $where;
    }

    public function relBalanco($set_codigo, $data_inicial = FALSE, $data_final = FALSE, $psi, $portaria = FALSE)
    {

        $subselect = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("prod2" => "produto"), array("pro_codigo",
                "(
				(SELECT SUM(COALESCE(sal.sal_qtde,0)) FROM saldo sal WHERE sal.pro_codigo = prod2.pro_codigo AND sal.set_codigo = $set_codigo)
				-
                (COALESCE(SUM(CASE WHEN (mov.mov_tipo in ('E','T') AND mov.set_entrada = $set_codigo) THEN ite.ite_quantidade ELSE 0 END),0))
                +
                (COALESCE(SUM(CASE WHEN (mov.mov_tipo in ('S','T') AND mov.set_saida = $set_codigo) THEN ite.ite_quantidade ELSE 0 END),0))
                ) as saldo_inicial"
            ))
            ->joinLeft(array("psi" => "psicotropicos"), "psi.psico_codigo=prod2.psico_codigo", array("psico_nome"))
            ->join(array("ite" => "itens_movimento"), "ite.pro_codigo=prod2.pro_codigo", "")
            ->join(array("mov" => "movimento"), "ite.mov_codigo=mov.mov_codigo", "")
            ->where("mov.set_entrada = '" . $set_codigo . "' OR mov.set_saida = '" . $set_codigo . "'")
            ->group(array("prod2.pro_codigo", "prod2.pro_nome", "psi.psico_nome"))
            ->order("pro_nome");

        if ($data_inicial)
            $subselect->where("mov.mov_data >= '$data_inicial'");

        if ($data_inicial) {
            $datai = " AND mov_data >= '$data_inicial'";
            $dataiSaldo = " AND sal_data >= '$data_inicial'";
            $datafSaldoInicial = "AND mov_data < '$data_inicial'";
        }
        if ($data_final) {
            $dataf = " AND mov_data <= '$data_final'";
            $datafSaldo = " AND sal_data <= '$data_final'";
        } else {
            $datafSaldoInicial = " AND mov_data <= 'NOW()'";
        }
        // Valida??es Psicotropicos
        if ($psi == "s") {
            $psicotropicos = " AND psico_codigo is not null";
        }
        // Valida??es de Data, Data Inicial e Final L?gica Antiga
        // Consulta SQL
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("pro" => "produto"), array("pro_nome",
                "pro_codigo_dcb",
                "pro_descricao_dcb",
                "pro_apresentacao_concentracao",
                "(SELECT
                    SUM(ite_quantidade)
                 FROM
                    produto p2
                 JOIN
                    itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                 JOIN
                    movimento m2 ON m2.mov_codigo = i2.mov_codigo
                 WHERE
                     1=1
                     $psicotropicos
                     $datai
                     $dataf
                     AND (mov_tipo = 'E' or mov_tipo = 'T')
                     AND set_entrada = $set_codigo
                     AND pro.pro_codigo = p2.pro_codigo) as entrada",
                "(SELECT
                    SUM(ite_quantidade)
                FROM
                    produto p2
                JOIN
                    itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                JOIN
                    movimento m2 ON i2.mov_codigo = m2.mov_codigo
                WHERE
                    1=1
                    $psicotropicos
                    $datai
                    $dataf
                    AND (mov_tipo = 'S' OR mov_tipo = 'T')
                    AND set_saida = $set_codigo
                    AND pro.pro_codigo = p2.pro_codigo
                    AND (mov_saida <> 'R' AND mov_saida <> 'S-PE')) as saida",
                "(SELECT
                    SUM(ite_quantidade)
                FROM
                    produto p2
                JOIN
                    itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                JOIN
                    movimento m2 ON i2.mov_codigo = m2.mov_codigo
                WHERE
                    1=1
                    $psicotropicos
                    $datai
                    $dataf
                    AND (mov_tipo = 'S')
                    AND (mov_saida = 'R' or mov_saida = 'S-PE')
                    AND set_saida = $set_codigo
                    AND pro.pro_codigo = p2.pro_codigo) as perda"))
            ->joinLeft(array("ite" => "itens_movimento"), "ite.pro_codigo=pro.pro_codigo", "")
            ->joinLeft(array("mov" => "movimento"), "mov.mov_codigo=ite.mov_codigo", "")
            ->joinLeft(array("f" => "fornecedor"), "f.for_codigo=mov.for_codigo", "")
            ->joinLeft(array("set" => "setor"), "set.set_codigo=mov.set_entrada", "")
            ->joinLeft(array("psi" => "psicotropicos"), "psi.psico_codigo=pro.psico_codigo", array("psico_nome"))
            ->join(array("saldo" => $subselect), "saldo.pro_codigo=pro.pro_codigo", "saldo_inicial")
            ->where("mov.set_entrada= $set_codigo OR mov.set_saida = $set_codigo")
            ->group(array("for_nome", "pro_nome", "for_cnpj", "set.set_codigo", "pro.pro_codigo", "mov_data", "psi.psico_nome", "saldo_inicial"))
            ->order("pro_nome");
        if ($psi == "s") {
            $where->where("pro.psico_codigo is not null");
        }

        if ($data_inicial)
            $where->where("mov.mov_data>= ?", $data_inicial);

        if ($data_final)
            $where->where("mov.mov_data <= ?", $data_final);
        if ($portaria)
            $where->where("pro.psico_codigo in ($portaria)");

        // echo "<pre>".print_r($where,1); exit;
        //die($where->__toString());
        //return $where;
        //die($where);
        return $this->fetchAll($where);
    }


    public function relSaidasTransferencias($set_codigo, $data_inicial = FALSE, $data_final = FALSE, $saida = FALSE, $perda = FALSE, $transferencia = FALSE, $set_codigo_destino = FALSE)
    {
        if ($data_inicial) {
            $datai = " AND m2.mov_data >= '$data_inicial'";

        }
        if ($data_final) {
            $dataf = " AND m2.mov_data <= '$data_final'";
        }

        // Valida??es de Data, Data Inicial e Final L?gica Antiga
        // Consulta SQL


        $dados = array("pro_nome",
            "(SELECT ROUND(AVG(ite2.ite_vlrunit),4) "
            . "FROM itens_movimento ite2 "
            . "INNER JOIN movimento mov2 ON mov2.mov_codigo=ite2.mov_codigo "
            . "WHERE ite2.pro_codigo=ite.pro_codigo AND ite2.ite_lote=ite.ite_lote AND mov2.mov_tipo='E') as custo_unitario");

        if ($saida) {
            array_push($dados, "(SELECT
                                                    SUM(ite_quantidade)
                                                FROM
                                                    produto p2
                                                JOIN
                                                    itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                                                JOIN
                                                    movimento m2 ON i2.mov_codigo = m2.mov_codigo
                                                WHERE
                                                    1=1
                                                    $datai
                                                    $dataf
                                                    AND (mov_tipo = 'S')
                                                    AND set_saida = $set_codigo
                                                    AND ite.ite_lote = i2.ite_lote
                                                    AND pro.pro_codigo = p2.pro_codigo
                                                    AND (mov_saida <> 'R' AND mov_saida <> 'S-PE' AND mov_saida <> 'S-TR')) as saida");
        }

        if ($perda) {
            array_push($dados, "(SELECT
                                                    SUM(ite_quantidade)
                                                FROM
                                                    produto p2
                                                JOIN
                                                    itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                                                JOIN
                                                    movimento m2 ON i2.mov_codigo = m2.mov_codigo
                                                WHERE
                                                    1=1
                                                    $datai
                                                    $dataf
                                                    AND (mov_tipo = 'S')
                                                    AND ite.ite_lote = i2.ite_lote
                                                    AND (mov_saida = 'R' or mov_saida = 'S-PE')
                                                    AND (mov_saida <> 'S-TR')
                                                    AND set_saida = $set_codigo
                                                    AND pro.pro_codigo = p2.pro_codigo) as perda");
        }
        if ($transferencia) {
            array_push($dados, "(SELECT
                                                    SUM(ite_quantidade)
                                                FROM
                                                    produto p2
                                                JOIN
                                                    itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                                                JOIN
                                                    movimento m2 ON i2.mov_codigo = m2.mov_codigo
                                                LEFT JOIN
                                                    requisicao_materiais req ON m2.req_codigo = req.rem_codigo
                                                WHERE
                                                    1=1
                                                    $datai
                                                    $dataf
                                                    AND ((m2.mov_tipo = 'T'" . ($set_codigo_destino ? " AND m2.set_entrada = $set_codigo_destino" : "") . ") OR (m2.mov_tipo = 'S' AND m2.mov_saida = 'S-TR'" . ($set_codigo_destino ? " AND req.set_codigo_req = $set_codigo_destino" : "") . "))
                                                    AND ite.ite_lote = i2.ite_lote
                                                    AND m2.set_saida = $set_codigo
                                                    AND pro.pro_codigo = p2.pro_codigo
                                                    AND (m2.mov_saida <> 'R' AND m2.mov_saida <> 'S-PE')) as transferencias");
        }


        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("pro" => "produto"), $dados)
            ->joinLeft(array("ite" => "itens_movimento"), "ite.pro_codigo=pro.pro_codigo", array("ite_lote"))
            ->joinLeft(array("mov" => "movimento"), "mov.mov_codigo=ite.mov_codigo", "")
            ->joinLeft(array("set" => "setor"), "set.set_codigo=mov.set_saida", "")
            ->group(array("pro_nome", "set.set_codigo", "pro.pro_codigo", "mov_data", "ite.pro_codigo", "ite.ite_lote"))
            ->order("pro_nome");

        if ($data_inicial)
            $where->where("mov.mov_data>= ?", $data_inicial);

        if ($data_final)
            $where->where("mov.mov_data <= ?", $data_final);
//        die($where);
        return $this->fetchAll($where);
    }

    public function relEntradas($set_codigo, $data_inicial = FALSE, $data_final = FALSE, $for_codigo = FALSE)
    {
        if ($data_inicial) {
            $datai = " AND mov_data >= '$data_inicial'";
            $dataiSaldo = " AND sal_data >= '$data_inicial'";
            $datafSaldoInicial = "AND mov_data < '$data_inicial'";
        }
        if ($data_final) {
            $dataf = " AND mov_data <= '$data_final'";
            $datafSaldo = " AND sal_data <= '$data_final'";
        } else {
            $datafSaldoInicial = " AND mov_data <= 'NOW()'";
        }
        // Valida??es de Data, Data Inicial e Final L?gica Antiga
        // Consulta SQL

        $dados = array("pro_nome", "pro_horus", "pro_horus_tipo",
            "(SELECT ROUND(AVG(ite2.ite_vlrunit),4) "
            . "FROM itens_movimento ite2 "
            . "INNER JOIN movimento mov2 ON mov2.mov_codigo=ite2.mov_codigo "
            . "WHERE ite2.pro_codigo=ite.pro_codigo AND ite2.ite_lote=ite.ite_lote AND mov2.mov_tipo='E') as custo_unitario",
            "(SELECT
                    SUM(ite_quantidade)
                FROM
                    produto p2
                JOIN
                    itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                JOIN
                    movimento m2 ON i2.mov_codigo = m2.mov_codigo
                WHERE
                    1=1
                    $psicotropicos
                    $datai
                    $dataf
                    AND (mov_tipo = 'E')
                    AND ite.ite_lote = i2.ite_lote
                    AND pro.pro_codigo = p2.pro_codigo) as entrada");


        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->distinct()
            ->from(array("pro" => "produto"), $dados)
            ->joinLeft(array("ite" => "itens_movimento"), "ite.pro_codigo=pro.pro_codigo", array("ite_lote"))
            ->joinLeft(array("mov" => "movimento"), "mov.mov_codigo=ite.mov_codigo", array("mov_nr_nota", "mov_codigo"))
            ->join(array("f" => "fornecedor"), "f.for_codigo=mov.for_codigo", array("for_nome", "for_cnpj"))
            ->joinLeft(array("set" => "setor"), "set.set_codigo=mov.set_entrada", "")
            ->where("mov.set_entrada = $set_codigo")
            ->group(array("for_nome", "mov_nr_nota","mov.mov_codigo", "for_cnpj", "pro_nome", "set.set_codigo", "pro.pro_codigo", "mov_data", "ite.pro_codigo", "ite.ite_lote"))
            ->order(array("for_nome", "pro_nome"));

        if ($data_inicial)
            $where->where("mov.mov_data>= ?", $data_inicial);

        if ($data_final)
            $where->where("mov.mov_data <= ?", $data_final);

        if ($for_codigo)
            $where->where("f.for_codigo = ?", $for_codigo);

//        die($where);
        return $this->fetchAll($where);
    }

    public function relEstoquePsicotropico(&$dados, $set_codigo)
    {
        $dados->title = "Estoque de Psicotrópicos";
        $dados->params = serialize($_POST);
        $dados->config = array(
            "th" => array("pro_nome" => "Medicamento", "sal_lote" => "Lote", "sal_validade" => "Validade", "sal_qtde" => "Quant.", "psico_nome" => "Portaria"),
            "formato" => array("sal_validade" => "data", "sal_qtde" => "num")
        );
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("sal" => "saldo"), array("sal_lote", "sal_validade", "sal_qtde"))
            ->join(array("pro" => "produto"), "pro.pro_codigo=sal.pro_codigo", array("pro_nome", "pro_horus", "pro_horus_tipo"))
            ->join(array("psico" => "psicotropicos"), "psico.psico_codigo=pro.psico_codigo", "psico_nome")
            ->where("sal.sal_qtde > 0")
            ->where("sal.set_codigo=?", $set_codigo)
            ->order("pro_nome");
        return $where;
    }

    public function relTransferencias(&$dados, $set_codigo, $data_inicial = FALSE, $data_final = FALSE)
    {
        $dados->title = "Transfer?ncias por fornecedores";
        $dados->params = serialize($_POST);
        $dados->config = array(
            "th" => array("set_nome" => "Setor de Destino", "mov_data" => "Data", "pro_nome" => "Produto", "ite_quantidade" => "Quantidade"),
            "formato" => array("ite_quantidade" => "num", "mov_data" => "data")
        );

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("mov" => "movimento"), "mov_data")
            ->join(array("ite" => "itens_movimento"), "ite.mov_codigo=mov.mov_codigo", "ite_quantidade")
            ->join(array("pro" => "produto"), "ite.pro_codigo=pro.pro_codigo", array("pro_nome"))
            ->join(array("set" => "setor"), "set.set_codigo=mov.set_entrada", "set_nome")
            ->where("mov.mov_tipo = 'T'")
            ->where("mov.set_saida=?", $set_codigo)
            ->order(array("set_nome", "mov_data", "pro_nome"));

        if ($data_inicial)
            $where->where("mov.mov_data>= ?", $data_inicial);

        if ($data_final)
            $where->where("mov.mov_data <= ?", $data_final);
        //die($where);
        return $where;
    }

    public function relAnvisa(&$dados, $set_codigo, $data_inicial = FALSE, $data_final = FALSE, $pro_codigo = FALSE, $portaria = FALSE)
    {
        $dados->title = "Relatório Anvisa";
        $dados->params = serialize($_POST);

        if ($data_inicial)
            $whereIni = " AND m2.mov_data >= '$data_inicial'";

        if ($data_final)
            $whereFim = " AND m2.mov_data <= '$data_final'";

        if ($set_codigo) {
            $whereSet = " AND m2.set_saida = $set_codigo";
        }
        if ($pro_codigo) {
            $wherePro = " AND p2.pro_codigo = $pro_codigo";
        } else {
            $wherePro = " AND i2.pro_codigo = p.pro_codigo";
        }

        $dados->config = array(
            "th" => array("mov_data" => "Data", "usu_nome" => "Nome", "pro_nome" => "Produto", "psico_nome" => "Portaria", "ite_quantidade" => "Quantidade", "set_nome" => "Setor", "usr_nome" => "Médico", "total" => "Total"),
            "formato" => array("ite_quantidade" => "num", "mov_data" => "data", "total" => "num")
        );
        $where = "SELECT  psi.psico_nome,u.usu_nome,
				  p.pro_nome, pro_horus, pro_horus_tipo,
				  ite_quantidade,
				  m.mov_data,
				  set_nome,
				  CASE
				   when
					usr_nome is not null
				   then
					usr_nome
				   when
					med_nome is not null
				   then
					med_nome
				   else
					'Médico Desconhecido'
				  END usr_nome ,

                                  (SELECT SUM(ite_quantidade)
                                     FROM movimento m2
                                     JOIN itens_movimento i2
                                       ON m2.mov_codigo =i2.mov_codigo
                                     JOIN usuario u2
                                       ON u2.usu_codigo = m2.usu_codigo
                                     JOIN produto p2
                                       ON p2.pro_codigo = i2.pro_codigo
                                    WHERE mov_tipo = 'S' and
                                            u2.usu_codigo is not null
                                            and p2.psico_codigo is not null
                                            $whereIni $whereFim $whereSet $wherePro) as total
		  from movimento m
		  join itens_movimento i
		   on i.mov_codigo = m.mov_codigo
		  join setor s
			on s.set_codigo = m.set_saida
		  join usuario u
			on u.usu_codigo = m.usu_codigo
		  join produto p
			on p.pro_codigo = i.pro_codigo
		  left join usuarios us
			on us.usr_codigo = m.med_codigo_interno
                  left join psicotropicos as psi
                        on psi.psico_codigo = p.psico_codigo
		  left join medico me
			on me.med_codigo = m.med_codigo_externo
			where mov_tipo = 'S' and
		u.usu_codigo is not null
		and p.psico_codigo is not null";

        $where .= " AND pro_situacao <> ''";

        if ($data_inicial)
            $where .= " AND m.mov_data >= '$data_inicial'";

        if ($data_final)
            $where .= " AND m.mov_data <= '$data_final'";

        if ($set_codigo) {
            $where .= " AND m.set_saida = $set_codigo";
        }
        if ($portaria) {
            $where .= " AND psi.psico_codigo in ($portaria)";
        }
        if ($pro_codigo) {
            $where .= " AND p.pro_codigo = $pro_codigo ORDER BY m.mov_data,pro_nome";
        } else {
            $where .= " ORDER BY m.mov_data ,pro_nome";
        }


        return $where;
    }

    public function dadosProdRelAnalitico($set_codigo = FALSE, $data_inicial = FALSE, $data_final = FALSE, $pro_codigo = FALSE, $portaria = FALSE)
    {
        // Validacoes de buscas
        if ($data_inicial) {
            $dataImov = "AND m2.mov_data >= '$data_inicial'";
            $datafSaldoInicial = "AND mov_data < '$data_inicial'";
        }
        if ($data_final) {
            $datafImov = "AND m2.mov_data <= '$data_final'";
        }
        if ($data_inicial == "" && $data_final == "") {
            //echo "BVVV";
            // Pega a data do ?ltimo movimento do setor e coloca saldo
            $datafSaldoInicial = "AND mov_data < (SELECT
                                                        mov_data
                                                      FROM
                                                        movimento AS mov
                                                      INNER JOIN
                                                        itens_movimento AS ite ON mov.mov_codigo=ite.mov_codigo
                                                      WHERE
                                                        (set_entrada = $set_codigo OR set_saida = $set_codigo) AND
                                                        (ite.pro_codigo = p2.pro_codigo)
                                                      ORDER BY
                                                        mov_data ASC
                                                        LIMIT 1)";
        }

        $subselect = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("prod2" => "produto"), array("pro_codigo",
                "(
				(SELECT SUM(COALESCE(sal.sal_qtde,0)) FROM saldo sal WHERE sal.pro_codigo = prod2.pro_codigo AND sal.set_codigo = $set_codigo)
				-
                (COALESCE(SUM(CASE WHEN (mov.mov_tipo in ('E','T') AND mov.set_entrada = $set_codigo) THEN ite.ite_quantidade ELSE 0 END),0))
                +
                (COALESCE(SUM(CASE WHEN (mov.mov_tipo in ('S','T') AND mov.set_saida = $set_codigo) THEN ite.ite_quantidade ELSE 0 END),0))
                ) as saldo_inicial"
            ))
            ->join(array("psi" => "psicotropicos"), "psi.psico_codigo=prod2.psico_codigo", array("psico_nome"))
            ->join(array("ite" => "itens_movimento"), "ite.pro_codigo=prod2.pro_codigo", "")
            ->join(array("mov" => "movimento"), "ite.mov_codigo=mov.mov_codigo", "")
            ->where("mov.set_entrada = '" . $set_codigo . "' OR mov.set_saida = '" . $set_codigo . "'")
            ->group(array("prod2.pro_codigo", "prod2.pro_nome", "psi.psico_nome"))
            ->order("pro_nome");

        if ($data_inicial)
            $subselect->where("mov.mov_data >= '$data_inicial'");


        $sql = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("prod" => "produto"), array("pro_codigo", "pro_nome",

                "COALESCE(SUM(CASE WHEN (mov.mov_tipo in ('E','T') AND mov.set_entrada = $set_codigo) THEN ite.ite_quantidade ELSE 0 END),0)
				as saldo_entrada",

                "COALESCE(SUM(CASE WHEN ((mov.mov_tipo in ('S','T') AND mov.set_saida = $set_codigo) AND (mov_saida <> 'R' AND mov_saida <> 'S-PE')) THEN ite.ite_quantidade ELSE 0 END),0)
				as saldo_saida",

                "COALESCE(SUM(CASE WHEN ((mov.mov_tipo = 'S') AND ((mov_saida = 'S-PE' OR mov_saida = 'R'))) THEN ite.ite_quantidade ELSE 0 END),0)
				as saldo_perda"
            ))
            ->join(array("psi" => "psicotropicos"), "psi.psico_codigo=prod.psico_codigo", array("psico_nome"))
            ->join(array("ite" => "itens_movimento"), "ite.pro_codigo=prod.pro_codigo", "")
            ->join(array("mov" => "movimento"), "ite.mov_codigo=mov.mov_codigo", "")
            ->join(array("saldo" => $subselect), "saldo.pro_codigo=prod.pro_codigo", "saldo_inicial")
            ->where("mov.set_entrada = '" . $set_codigo . "' OR mov.set_saida = '" . $set_codigo . "'")
            ->group(array("prod.pro_codigo", "prod.pro_nome", "psi.psico_nome", "saldo.saldo_inicial"))
            ->order("pro_nome");
        // Validacoes de buscas
        if ($data_inicial)
            $sql->where("mov.mov_data >= '$data_inicial'");
        if ($data_final)
            $sql->where("mov.mov_data <= '$data_final'");
        if ($pro_codigo)
            $sql->where("prod.pro_codigo =?", $pro_codigo);
        if ($portaria)
            $sql->where("psi.psico_codigo in ($portaria)");

        //die($sql);
        return $this->fetchAll($sql);
    }

    public function dadosPacienteRelAnalitico($pro_codigo, $data_inicial, $data_final, $set_codigo, $portaria = FALSE)
    {
        $sql = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("mov" => "movimento"), array("usu_codigo", "mov_codigo", "mov_data", "set_entrada", "set_saida", "mov_tipo AS mov_tipo_imp", "mov_saida", "mov_nr_nota", "mov_observacao"))
            ->join(array("ite" => "itens_movimento"), "mov.mov_codigo=ite.mov_codigo", array("ite_quantidade", "ite_cod_receita"))
            ->joinLeft(array("usu" => "usuario"), "mov.usu_codigo=usu.usu_codigo", array("usu_prontuario", "usu_nome"))
            ->joinLeft(array("usrs" => "usuarios"), "mov.usr_codigo=usrs.usr_codigo", array("usr_nome"))
            ->joinLeft(array("forn" => "fornecedor"), "mov.for_codigo=forn.for_codigo", array("for_nome"))
            ->joinLeft(array("pro" => "produto"), "ite.pro_codigo=pro.pro_codigo")
            ->joinLeft(array("psi" => "psicotropicos"), "psi.psico_codigo=pro.psico_codigo", array("psico_nome"))
            ->where("ite.pro_codigo =?", $pro_codigo)
            ->where("set_entrada = $set_codigo OR set_saida = $set_codigo");
        // Validacoes de buscas
        if ($data_inicial)
            $sql->where("mov.mov_data >= '$data_inicial'");
        if ($data_final)
            $sql->where("mov.mov_data <= '$data_final'");

        if ($portaria)
            $sql->where("psi.psico_codigo = '$portaria'");

        $sql->order(array("mov_data", "ite_codigo", "usu_nome"));

        //die($sql);
        return $this->fetchAll($sql);
    }

    /**
     * Atualiza todas as tabelas do sistema que est?o vinculadas a um paciente duplicado para um ?nico paciente
     * depois remove-os.
     * @param int $correto
     * @param array $duplicados
     * @return array quantos registros foram atualizados e quantos foram removidos por tabela
     */
    public function removerDuplicacoes($correto, $duplicados)
    {
        Zend_Registry::get("logger")->log(array($correto, $duplicados), Zend_Log::INFO);

        $out = array();
        $tbCar = new Application_Model_Carteirinha();
        $tbCot = new Application_Model_CotaSetor();
        $tbEve = new Application_Model_EventoProduto();
        $tbInv = new Application_Model_InventarioProduto();
        $tbIte = new Application_Model_MovimentoItens();
        $tbiReq = new Application_Model_ItensRequisicao();
        $tbLei = new Application_Model_LeitoGradeItens();
        $tbSet = new Application_Model_ProdutoSetor();
        $tbPro = new Application_Model_ProgramaProduto();
        $tbReq = new Application_Model_RequisicaoCompraProduto();
        $tbSal = new Application_Model_Saldo();
        $tbVac = new Application_Model_VacinaUsuario();
        $tbIteBkp = new Application_Model_ItensMovimentoBkp();


        $this->getAdapter()->beginTransaction();

        try {
            $out['car'] = $tbCar->atualizarPro($duplicados, $correto);
            $out['cot'] = $tbCot->atualizarPro($duplicados, $correto);
            $out['eve'] = $tbEve->atualizarPro($duplicados, $correto);
            $out['inv'] = $tbInv->atualizarPro($duplicados, $correto);
            $out['ite'] = $tbIte->atualizarPro($duplicados, $correto);
            $out['ireq'] = $tbiReq->atualizarPro($duplicados, $correto);
            $out['lei'] = $tbLei->atualizarPro($duplicados, $correto);
            $out['set'] = $tbSet->atualizarPro($duplicados, $correto);
            $out['pro'] = $tbPro->atualizarPro($duplicados, $correto);
            $out['req'] = $tbReq->atualizarPro($duplicados, $correto);
            $out['sal'] = $tbSal->atualizarPro($duplicados, $correto);
            $out['vac'] = $tbVac->atualizarPro($duplicados, $correto);
            $out['bkp'] = $tbIteBkp->atualizarPro($duplicados, $correto);

            $this->getAdapter()->commit();
        } catch (Exception $e) {
            $this->getAdapter()->rollBack();
            Zend_Registry::get("logger")->log($e->getMessage(), Zend_Log::INFO);
            return false;
        }

        $removidos = $this->remover($duplicados);


        return array(array_sum($out), $removidos);
    }

    /**
     * Recebe um array de usu_codigo e remove todos
     * @param array $usu_codigo
     * @return int N?mero de linhas removidas
     */
    public function remover($pro_codigo)
    {
        $where = $this->select()
            ->where("pro_codigo IN (?)", $pro_codigo)
            ->getPart(Zend_Db_Table_Select::WHERE);
        $where = $where[0];
        return $this->delete($where);
    }

    public function getProduto($pro_codigo = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"))
            ->join(array("umed" => "unidmedida"), "umed.umed_codigo=pro.umed_codigo", "umed_nome")
            ->where("pro_codigo=$pro_codigo");
//		die($where);
        return $this->fetchRow($where);
    }

    public function getTodosProdutos()
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"), array("pro_codigo"))
            ->order("pro_codigo");
//		die($where);
        return $this->fetchAll($where);
    }

  public function listProdutos()
  {
    $where = $this->select(FALSE)
      ->setIntegrityCheck(FALSE)
      ->from(array("pro" => "produto"), array("pro_codigo", "pro_nome", "pro_situacao"))
      ->order("pro_codigo DESC")
      ->limit(15);
//		die($where);
    return $this->fetchAll($where);
  }

    public function getProdutoComEstoque($pro_codigo = FALSE, $set_codigo = FALSE, $ite_codbarras = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"))
            ->join(array("umed" => "unidmedida"), "umed.umed_codigo=pro.umed_codigo", "umed_nome")
            ->join(array("sal" => "saldo"), "sal.pro_codigo=pro.pro_codigo", "")
            ->joinLeft(array("psico" => "psicotropicos"), "psico.psico_codigo=pro.psico_codigo", "psico_exige_codigo")
            ->where("sal_qtde > 0")
            ->where("set_codigo = $set_codigo")
            ->where("sal.pro_codigo=$pro_codigo");

        if ($ite_codbarras) {
            $where->where("EXISTS(SELECT itm.ite_codbarras FROM itens_movimento itm WHERE itm.pro_codigo=$pro_codigo AND itm.ite_codbarras='$ite_codbarras' AND sal.sal_lote=itm.ite_lote LIMIT 1)");
        }


        //die($where);
        return $this->fetchRow($where);
    }

    public function getProdutoComEstoqueComCodBarras($pro_codigo = FALSE, $set_codigo = FALSE, $ite_codbarras = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto", array()))
            ->joinLeft(array("psico" => "psicotropicos"), "psico.psico_codigo=pro.psico_codigo", "psico_exige_codigo")
            ->join(array("umed" => "unidmedida"), "umed.umed_codigo=pro.umed_codigo", "umed.umed_nome")
            ->join(array("proset" => "produto_setor"), "proset.pro_codigo=pro.pro_codigo", "")
            ->join(array("sal" => "saldo"), "sal.pro_codigo=pro.pro_codigo AND sal.set_codigo = proset.set_codigo", "SUM(sal.sal_qtde) AS quantidade")
            ->join(array("ite" => "itens_movimento"), " sal.pro_codigo=ite.pro_codigo AND sal.sal_lote=ite.ite_lote", "ite_codbarras")
            ->where("sal_qtde > 0")
            ->where("proset.set_codigo = $set_codigo")
            ->group(array("pro.pro_codigo", "umed.umed_nome", "ite.ite_codbarras", "psico.psico_exige_codigo"));
        if ($pro_codigo) {
            $where->where("sal.pro_codigo=$pro_codigo");
        }

        if ($ite_codbarras) {
            $where->where("EXISTS(SELECT TRUE FROM itens_movimento itm WHERE itm.ite_codbarras='$ite_codbarras' AND sal.pro_codigo=itm.pro_codigo AND sal.sal_lote=itm.ite_lote LIMIT 1)");
        }

//		die($where);
        return $this->fetchRow($where);
    }

    public function getRankingProdutos($set_codigo = FALSE, $quantidade = FALSE, $data_inicial = FALSE, $data_final = FALSE, $pros_codigo = FALSE)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("p" => "produto"), array("pro_nome", "pro_horus", "pro_horus_tipo"))
            ->join(array("im" => "itens_movimento"), "im.pro_codigo=p.pro_codigo", array("quantidade" => "sum(ite_quantidade)"))
            ->join(array("m" => "movimento"), "m.mov_codigo=im.mov_codigo", "")
            ->where("mov_tipo='S'")
            ->group(array("p.pro_nome", "p.pro_horus", "p.pro_horus_tipo"))
            ->having("sum(ite_quantidade) > 0")
            ->order("2 desc");

        if ($set_codigo)
            $where->where("set_saida=$set_codigo");

        if ($quantidade)
            $where->limit($quantidade);

        if ($data_inicial)
            $where->where("mov_data >= '$data_inicial'");

        if ($data_final)
            $where->where("mov_data <= '$data_final'");

        if ($pros_codigo)
            $where->where("pros_codigo=$pros_codigo");


        return $this->fetchAll($where);
    }

    public function getUltimoRegistro($term = NULL)
    {
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("for" => "produto"), array("pro_codigo", "pro_nome", "pro_horus", "pro_horus_tipo"))
            ->where("retira_acentos(pro_nome) ilike retira_acentos('%$term%')")
            ->order("pro_codigo desc")
            ->limit(1);
        return $this->fetchRow($where)->toArray();;
    }

    public function produtosSemCatMat($pro_codigo = FALSE)
    {
        $sql = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"), array("pro_codigo", "pro_nome"))
            ->where("pro_horus is null OR (length(pro_horus)<14)")
            ->where("pro_situacao = 'A'");
        if ($pro_codigo) {
            $sql->where("pro_codigo = $pro_codigo");
        }

        $sql->order("pro_nome");
        return $this->fetchAll($sql);
    }

    public function produtosSemCatMatPorNome($term = FALSE)
    {
        $sql = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"), array("pro_codigo", "pro_nome"))
            ->where("pro_horus is null OR (length(pro_horus)<14)")
            ->where("pro_situacao = 'A'");
        if ($term) {
            $sql->where("pro_nome ilike '%$term%'");
        }
        $sql->order("pro_nome");
        return $this->fetchAll($sql);
    }
    public function buscarVacina($term = false)
    {

        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"), array("pro_codigo", "pro_nome"))
            ->join(array("gru" => "grupo"), "pro.gru_codigo = gru.gru_codigo");
        if ($term)
            $where->where("pro_nome ilike '%$term%'");
        $where->where("gru.gru_nome ilike '%VACINA%'")
            ->order("pro_nome");

        $all = $this->fetchAll($where);

        $out = array();
        foreach ($all as $pro) {
            $out [] = array(
                "id" => $pro->pro_codigo,
                "label" => $pro->pro_nome,
                "data" => array("pro_codigo" => $pro->pro_codigo, "pro_nome" => $pro->pro_nome)
            );
        }

        if (!count($out)) {
            $out [] = array(
                "id" => 0,
                "label" => "Nenhum item encontrado",
                "data" => array()
            );
        }
        return $out;
    }

    public function buscarMaterialHospitalar($term = FALSE, $set_codigo, $limite = FALSE, $movimento = FALSE)
    {
        if(!$set_codigo){
            return false;
        }
        $where = $this->select(FALSE)
            ->setIntegrityCheck(FALSE)
            ->from(array("pro" => "produto"))
            ->joinLeft(array("umed" => "unidmedida"), "pro.umed_codigo = umed.umed_codigo", array("umed_nome"))
            ->join(array("sal" => "saldo"), "sal.pro_codigo=pro.pro_codigo", array("sal_qtde", "sal_lote", "sal_validade"))
            ->where("pro_situacao='A'")
            ->where("sal.sal_qtde > 0")
            ->where("set_codigo = $set_codigo")
            ->where("pro.gru_codigo IN (SELECT gru_codigo FROM grupo WHERE gru_nome ilike '%MATERIAL HOSPITALAR%')")
            ->order("pro_nome");

        if ($term)
            $where->where("pro.pro_nome ilike '%$term%'");

        if ($term)
            $where->where("pro_nome ilike '%$term%'");

        if ($limite)
            $where->limit($limite);

        $all = $this->fetchAll($where);

        $out = array();
        foreach ($all as $pro) {
            $out [] = array(
                "id" => $pro->pro_codigo,
                "label" => $pro->pro_nome,
                "data" => array("pro_codigo" => $pro->pro_codigo, "pro_nome" => $pro->pro_nome, "umed_nome" => $pro->umed_nome, "sal_qtde" => $pro->sal_qtde, "sal_lote" => $pro->sal_lote, "sal_validade" => $pro->sal_validade)
            );
        }

        if (!count($out)) {
            $out [] = array(
                "id" => 0,
                "label" => "Nenhum item encontrado",
                "data" => array()
            );
        }
        return $out;
    }

    public function getImunobiologico($pro_codigo){
        $where = $this->select()
                    ->from(["pro"=>"produto"], "tvi_codigo")
                    ->where("pro.pro_codigo = $pro_codigo");
        return $this->fetchRow($where);
    }

  public function pesquisar($dados) {
    $where = $this->select(true)
                  ->where("pro_nome ilike '%$dados%'")
                  ->orWhere("pro_codigo = ?", (int) $dados);

    $where->limit(15);
    return $this->fetchAll($where);
  }

  public function editar($pro_codigo) {
    return $this->fetchRow("pro_codigo=$pro_codigo");
  }

  public function getVinculosPorProduto($pro_codigo)
  {
    $sql = $this->select()
      ->setIntegrityCheck(FALSE)
      ->from(["prset" => "produto_setor"])
      ->join(["set" => "setor"], "prset.set_codigo = set.set_codigo", "set_nome")
      ->where("prset.pro_codigo = $pro_codigo")
      ->order("set.set_nome");
    return $this->fetchAll($sql);
  }

  public function getProdutosVencimento($id)
  {
    $arrayResult = [];
    $arrayResult[0] = [];
    $arrayResult[1] = [];

    //Checa setor logado
    $sqlSetor = $this->select()
      ->setIntegrityCheck(FALSE)
      ->from("logon", array("cod_setor"))
      ->where("id_login = $id")
      ->order("dt_entrada DESC");
     $setor = $this->fetchRow($sqlSetor)['cod_setor'];

    //Checa se o setor exibe alerta
    $sqlPossuiAlerta = $this->select()
      ->setIntegrityCheck(FALSE)
      ->from("setor", array("mostra_alerta_estoque", "qtd_dias_alerta"))
      ->where("set_codigo = $setor");
    $exibeAlerta = $this->fetchRow($sqlPossuiAlerta);

    if($exibeAlerta['mostra_alerta_estoque'] == 'S'){

      $diasParaAlerta = $exibeAlerta['qtd_dias_alerta'];
      $dataFinal = date('Y-m-d', mktime(0, 0, 0, date('m'), date('d') + $diasParaAlerta, date('Y')));

      $sqlVencimento = $this->select()
        ->setIntegrityCheck(FALSE)
        ->from(array("s" => "saldo"))
        ->join(array("p" => "produto"), "s.pro_codigo = p.pro_codigo", array('pro_nome'))
        ->where("p.pro_situacao is not null")
        ->where("s.set_codigo = $setor")
        ->where("s.sal_qtde > 0")
        ->where("sal_validade <= '$dataFinal'")
        ->order(array("sal_validade", "p.pro_nome"));

      $arrayResult[0] = $this->fetchAll($sqlVencimento)->toArray();

      $sqlEstoqueMinimo = $this->select()
                ->setIntegrityCheck(FALSE)
                ->from(array("sal" => "saldo"), array("SUM(sal_qtde) as sal_qtde"))
                ->join(array("pro" => "produto"), "pro.pro_codigo=sal.pro_codigo", array("pro_codigo","pro_nome"))
                ->join(array("proset" => "produto_setor"), "pro.pro_codigo=proset.pro_codigo", "prset_minimo")
                ->joinLeft(array("pros" => "produto_subgrupo"), "pro.pros_codigo=pros.pros_codigo", "pros_codigo")
                ->joinLeft(array("gru" => "grupo"), "pro.gru_codigo=gru.gru_codigo", "gru_codigo")
                ->where("sal.set_codigo=$setor ")
                ->where("proset.set_codigo=$setor ")
                ->where("proset.prset_minimo IS NOT NULL ")
                ->where("(select sum(sal1.sal_qtde) "
                      . "   from saldo sal1 "
                      . "   join produto pro1 on pro1.pro_codigo=sal1.pro_codigo "
                      . "   join produto_setor proset1 on pro1.pro_codigo=proset1.pro_codigo "
                      . "  where sal1.set_codigo=$setor and "
                      . "        proset1.set_codigo=$setor and "
                      . "        proset1.prset_minimo IS NOT NULL and "
                      . "        pro1.pro_codigo = pro.pro_codigo "
                      . ") <= proset.prset_minimo ")
                ->where("pro.pro_situacao = 'A'")
                ->group(array("pro_nome", "prset_minimo", "pro.pro_codigo", "pros.pros_codigo", "gru.gru_codigo"))
                ->order("pro_nome");

      $arrayResult[1] = $this->fetchAll($sqlEstoqueMinimo)->toArray();

      return $arrayResult;

    }else{
      return false;
    }
  }

  public function getProdutosDoSetor($setorId,$data){
    $sql = $this->select()
                ->setIntegrityCheck(FALSE)
                ->distinct(TRUE)
                ->from(array("p" => "produto"), array("CONCAT(pro_horus, pro_tipo) as nuProduto"))
                ->join(array("ps" => "produto_setor"), "ps.pro_codigo = p.pro_codigo", "")
                ->join(array("s" => "setor"), "s.set_codigo = ps.set_codigo", array("set_tipo_estab"))
                ->join(array("u" => "unidade"), "u.uni_codigo = s.uni_codigo", array("uni_cnes", "uni_codigo_ibge"))
                ->join(array("sal" => "saldo"), "sal.pro_codigo = p.pro_codigo and sal.set_codigo = s.set_codigo", array("sal_lote", "to_char(sal_validade, 'dd-mm-YYYY') as sal_validade",
                                      "
                                      (
                                      sal.sal_qtde -- saldo atual
                                      -
                                      (SELECT ROUND(COALESCE(SUM(ite_quantidade), 0))
                                                           FROM   produto p2
                                                           join itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                                                           join movimento m2 ON m2.mov_codigo = i2.mov_codigo
                                                          WHERE ( mov_tipo = 'E' OR mov_tipo = 'T' )
                                                            AND mov_entrada IS NOT NULL
                                                            AND mov_data >= '$data'
                                                            AND set_entrada = $setorId
                                                            AND sal.pro_codigo = p2.pro_codigo) -- saldo de entradas
                                      +
                                      (SELECT ROUND(COALESCE(SUM(ite_quantidade), 0))
                                                           FROM   produto p2
                                                           join itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                                                           join movimento m2 ON m2.mov_codigo = i2.mov_codigo
                                                          WHERE ( mov_tipo = 'S' OR mov_tipo = 'T' )
                                                            AND mov_saida IS NOT NULL
                                                            AND mov_data >= '$data'
                                                            AND set_saida = $setorId
                                                            AND sal.pro_codigo = p2.pro_codigo) -- salde de saídas
                                      ) AS posicao_estoque"))
                ->where("ps.set_codigo = $setorId")
                ->where("p.pro_situacao = 'A'")
                ->where("p.pro_horus is not null")
                ->where("p.pro_codigo is not null")
                ->where("p.gru_codigo is not null")
                ->where("p.pro_tipo is not null")
                ->where("s.set_tipo_estab is not null")
                ->where("s.set_tipo_estab in ('A','R','F')")
                ->where("u.uni_codigo_ibge is not null")
                ->where("sal.sal_lote is not null")
                ->where("sal.sal_validade is not null")
                ->where("((
                            sal.sal_qtde -- saldo atual
                            -
                            (SELECT ROUND(COALESCE(SUM(ite_quantidade), 0))
                                                 FROM   produto p2
                                                 join itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                                                 join movimento m2 ON m2.mov_codigo = i2.mov_codigo
                                                WHERE ( mov_tipo = 'E' OR mov_tipo = 'T' )
                                                  AND mov_entrada IS NOT NULL
                                                  AND mov_data >= '$data'
                                                  AND set_entrada = $setorId
                                                  AND sal.pro_codigo = p2.pro_codigo) -- saldo de entradas
                            +
                            (SELECT ROUND(COALESCE(SUM(ite_quantidade), 0))
                                                 FROM   produto p2
                                                 join itens_movimento i2 ON i2.pro_codigo = p2.pro_codigo
                                                 join movimento m2 ON m2.mov_codigo = i2.mov_codigo
                                                WHERE ( mov_tipo = 'S' OR mov_tipo = 'T' )
                                                  AND mov_saida IS NOT NULL
                                                  AND mov_data >= '$data'
                                                  AND set_saida = $setorId
                                                  AND sal.pro_codigo = p2.pro_codigo) -- salde de saídas
                            ) >= 0)");

    return $this->fetchAll($sql);
  }
}
