<?php

Zend_Loader::loadClass("Elotech_Db_Table_Abstract");

/**
 * Essa agenda é resposável pelo novo agendamento
 * @todo agendar exames e consultas, por quantidade (cota) e valor
 */
class Application_Model_CompraProduto extends Elotech_Db_Table_Abstract {

	protected $_name = 'compra_produto';
	protected $_primary = 'comp_codigo';
	/**
	 * Insert ou update na agenda, com seus filhos
	 * @param array $data dados do formulário
	 * @return int chave primária do registro inserido ou atualizado 
	 */
	public function salvar(array $data) {
		
		$this->valoresPadrao($data);
		$this->notEmpty(array("usu_codigo"), $data);
		$this->solicitanteInternoOuExterno($data);
		$this->peloMenosUm(array("med_codigo", "usr_codigo_medico"), $data);
		$this->emptyToUnset($data);		
		
		$itens = $data['itens'];
		unset($data['itens']);

		$this->getDefaultAdapter()->beginTransaction();
		try {
			$age_codigo = parent::salvar($data);

			$tbAgeI = new Application_Model_AgendaItens();
			$tbAgeI->salvarDoArray($itens, $age_codigo, $data);

			$this->getDefaultAdapter()->commit();
		} catch (Exception $exc) {
			$this->getDefaultAdapter()->rollBack();
			throw new Zend_Validate_Exception($exc->getMessage());
		}

		return $age_codigo;
	}

	/**
	 * Trabalha os dados recebidos e diz se o médico solicitante é interno ou externo
	 * @param array $data dados do formulário
	 */
	private function solicitanteInternoOuExterno(&$data) {
		if (!$data['interno']) { // 1=interno, 0=externo
			$data["med_codigo"] = $data["usr_codigo_medico"];
			unset($data["usr_codigo_medico"]);
		}

		unset($data['interno']);
	}

	/**
	 * Retorna os dados do agendamento
	 * @param int $age_codigo
	 * @return Zend_Db_Table_Rowset_Abstract 
	 */
	public function getAgendamento($age_codigo) {
		$where = $this->select(FALSE)
				->setIntegrityCheck(FALSE)
				->from(array("age" => "agenda"), "usu_codigo")
				->join(array("agei" => "agenda_itens"), "agei.age_codigo=age.age_codigo", "agei_data")
				->join(array("coni" => "convenio_itens"), "coni.coni_codigo=agei.coni_codigo", array("coni_valor"))
				->join(array("conv" => "convenio"), "conv.conv_codigo=coni.conv_codigo", "")
				->join(array("med" => "medico"), "med.med_codigo=conv.med_codigo", array("med_nome", "med_endereco")) // joinLeft + joinLeft(unidade)
				->joinLeft(array("cid" => "cidade"), "cid.cid_codigo=med.cid_codigo", "cid_nome")
				->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_nome", "proc_codigo_sus"))
				->where("age.age_codigo=?", $age_codigo)
				->order("proc_nome");

		//die($where->__toString());
		return $this->fetchAll($where);
	}

	/**
	 * Retorna quais as orientações para o agendamento informado
	 * @param int $age_codigo
	 * @return Zend_Db_Table_Rowset_Abstract 
	 */
	public function getOrientacoes($age_codigo) {
		$where = $this->select(FALSE)
				->setIntegrityCheck(FALSE)
				->distinct()
				->from(array("age" => "agenda"), "")
				->join(array("agei" => "agenda_itens"), "agei.age_codigo=age.age_codigo", "")
				->join(array("coni" => "convenio_itens"), "coni.coni_codigo=agei.coni_codigo", "")
				->join(array("proo" => "procedimento_orientacoes"), "proo.proc_codigo=coni.proc_codigo", "")
				->join(array("ori" => "orientacoes_exames"), "ori.ori_exa_codigo=proo.ori_exa_codigo", "ori_exa_orientacoes")
				->where("age.age_codigo=?", $age_codigo)
				->group("ori_exa_orientacoes")
		;

		return $this->fetchAll($where);
	}

	/**
	 * Valores padrão do insert/update
	 * @param array $data valores do insert
	 */
	private function valoresPadrao(&$data) {
		if (empty($data['age_data_insert'])) {
			$data['age_data_insert'] = date("Y-m-d H:i:s");
		}

		if (empty($data['usr_codigo'])) {
			$tbUsr = new Application_Model_Usuarios();
			$data['usr_codigo'] = $tbUsr->getUsrAtual()->usr_codigo; // pode gerar exception
		}
	}

	/**
	 * Retorna quantas vagas há entre as datas informadas
	 * @param array $coni_codigos lista de itens do convenio
	 */
	public function getVagas($coni_codigos, $data_inicial, $data_final) {
		$coni_codigos = (array) $coni_codigos;

		$tbConv = new Application_Model_Convenio();
		$tbConI = new Application_Model_ConvenioItens();

		$conv_codigo = $tbConI->find($coni_codigos[0])->current()->conv_codigo;

		// verificar se o local(L/H/U) atende aos sábados e domingos
		$sd = $tbConv->atendeSabadoEDomingo($conv_codigo);

		$tbFun = new Application_Model_Funcoes();
		$arrDatas = $tbFun->datasToArray($data_inicial, $data_final);
		$arrFinal = array();

		foreach ($coni_codigos as $coni_codigo) {
			$arrConi = array();
			$vagas = $this->vagas($coni_codigo, $data_inicial, $data_final);
			foreach ($vagas as $dia) {
				if ($dia->limite_dia < 0)
					$vagasRestantesDia = -1;
				else
					$vagasRestantesDia = $dia->limite_dia - $dia->agendado_dia;

				if ($dia->limite_mes < 0)
					$vagasRestantesMes = -1;
				else
					$vagasRestantesMes = $dia->limite_mes - $dia->agendado_mes;



				if (!$this->sabadoDomingoOuFeriado($dia->grad_dia, $sd))
					$arrConi[$dia->grad_dia] = 0; // não podem haver vagas nos sábados*, domingos* e feriados
				else
					$arrConi[$dia->grad_dia] = $this->vagasRestantes($vagasRestantesDia, $vagasRestantesMes);
			}

			if (count($arrConi) < count($arrDatas)) {
				$tbGram = new Application_Model_GradeMes();
				$tbGrad = new Application_Model_GradeDia();

				foreach ($arrDatas as $dia) {
					if (!array_key_exists($dia, $arrConi)) {
						if (!$this->sabadoDomingoOuFeriado($dia, $sd)) {
							$arrConi[$dia] = 0; // não podem haver vagas nos sábados*, domingos* e feriados
						} else {
							$limite_dia = $tbGrad->getCotaDia($coni_codigo, $dia);
							$limite_mes = $tbGram->getCotaMes($coni_codigo, $dia);
							$arrConi[$dia] = min(array($limite_dia, $limite_mes));
						}
					}
				}
			}
			ksort($arrConi);
			$arrFinal [$coni_codigo] = $arrConi;
		}

		return $arrFinal;
	}

	/**
	 * Retornar quantas vagas há por dia
	 * @param int $coni_codigo
	 * @param date $data_inicial
	 * @param date $data_final
	 * @return array 
	 */
	public function getTotalVagas($coni_codigo, $data_inicial, $data_final) {
		$arrConi = array();
		// vagas agendadas
		$vagas = $this->vagas($coni_codigo, $data_inicial, $data_final);
		foreach ($vagas as $dia) {
			$arrConi[$dia->grad_dia] = (object) $dia->toArray();
		}

		// vagas ainda não agendadas
		$tbFun = new Application_Model_Funcoes();
		$arrDatas = $tbFun->datasToArray($data_inicial, $data_final);

		if (count($arrConi) < count($arrDatas)) {
			$tbGram = new Application_Model_GradeMes();
			$tbGrad = new Application_Model_GradeDia();

			$aux = new stdClass();
			$aux->agendado_dia = 0;
			$aux->agendado_mes = 0;

			foreach ($arrDatas as $dia) {
				if (!array_key_exists($dia, $arrConi)) {
					$aux->grad_dia = $dia;
					$aux->coni_cota_mes = $tbGram->getCotaMes($coni_codigo, $dia);
					$aux->limite_mes = $aux->coni_cota_mes;
					$aux->coni_cota_dia = $tbGrad->getCotaDia($coni_codigo, $dia);
					$aux->limite_dia = $aux->coni_cota_dia;
					$arrConi[$dia] = $aux;
				}
			}

			ksort($arrConi);
		}

		return $arrConi;
	}

	/**
	 * Retornar quantas vagas há por dia, por coni_codigo
	 * @param array $coni_codigos
	 * @param date $data_inicial
	 * @param date $data_final
	 * @return array 
	 */
	public function getTotalVagasArr($coni_codigos, $data_inicial, $data_final) {
		$out = array();
		foreach ($coni_codigos as $coni_codigo)
			$out [$coni_codigo] = $this->getTotalVagas($coni_codigo, $data_inicial, $data_final);

		return $out;
	}

	/**
	 * Verifica se ha um limitador no dia ou mes
	 * @param int $dia
	 * @param int $mes 
	 */
	private function vagasRestantes($dia, $mes) {
		if ($dia < 0) {
			return $mes;
		} elseif ($dia == 0) {
			return 0;
		} else {
			if ($mes < 0)
				return $dia;

			return min(array($dia, $mes));
		}
	}

	/**
	 * Retorna quantas vagas foram usadas por dia, quantas há disponíveis por dia e quantas há disponível por mês
	 * @see SQL Anderson
	 * @see Zend_Db_Expr - http://stackoverflow.com/questions/6663473/zend-db-select-left-join-on-a-subselect
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function vagas($coni_codigo, $data_inicial=FALSE, $data_final=FALSE) {
		$sql1 = $this->select(FALSE)
				->setIntegrityCheck(FALSE)
				->from(array("age" => "agenda_itens"), "count(agei_codigo)") // quantos foram agendados por dia
				->where("agei_data=grad_dia")
				->where("age.coni_codigo=$coni_codigo");
		$sql1 = $this->select(FALSE)
				->setIntegrityCheck(FALSE)
				->from(array("age" => "agenda_itens"), "count(agei_codigo)") // quantos foram agendados por dia
				->where("agei_data=grad_dia")
				->where("age.coni_codigo=$coni_codigo");

		$sql2 = $this->select(FALSE)
				->setIntegrityCheck(FALSE)
				->from(array("gram" => "grade_mes"), array("gram_cota_mes")) // quantas vagas há no mês
				->where("to_char(gram_mes,'mm/yyyy') = to_char(grad_dia,'mm/yyyy')")
				->where("gram.coni_codigo=$coni_codigo");

		$sql3 = $this->select(FALSE)
				->setIntegrityCheck(FALSE)
				->from(array("age" => "agenda_itens"), "count(agei_codigo)") // quantos foram agendados no mês
				->where("to_char(agei_data,'mm/yyyy') = to_char(grad_dia,'mm/yyyy')")
				->where("age.coni_codigo=$coni_codigo");

		$sql4 = $this->select(FALSE)
				->setIntegrityCheck(FALSE)
				->from(array("gram" => "grade_mes"), array("gram_alterada")) // o mês foi alterado?
				->where("to_char(gram_mes,'mm/yyyy') = to_char(grad_dia,'mm/yyyy')")
				->where("gram.coni_codigo=$coni_codigo");

		$where = $this->select(FALSE)
				->setIntegrityCheck(FALSE)		  // quanto pode por dia
				->from(array("grad" => "grade_dia"), array("grad_dia", "limite_dia" => "grad_cota_dia", "grad_alterada"))
				->columns(array(
					"agendado_dia" => new Zend_Db_Expr('(' . $sql1->assemble() . ')'),
					"limite_mes" => new Zend_Db_Expr('(' . $sql2->assemble() . ')'),
					"agendado_mes" => new Zend_Db_Expr('(' . $sql3->assemble() . ')'),
					"gram_alterada" => new Zend_Db_Expr('(' . $sql4->assemble() . ')')
				))
				->join(array("coni" => "convenio_itens"), "coni.coni_codigo=grad.coni_codigo", array("coni_cota_mes", "coni_cota_dia"))
				->where("coni.coni_codigo=?", $coni_codigo)
				->order("grad_dia");

		if ($data_inicial)
			$where->where("grad_dia >= '$data_inicial'");

		if ($data_final)
			$where->where("grad_dia <= '$data_final'");

		//die($where->__toString());
		return $this->fetchAll($where);
	}

	/**
	 * Informa a data final, para montar o grid de agendamento
	 * Também valida a data inicial, impedindo data passada
	 * @param date $data_inicial
	 * @param bool $fimDoMes Verdadeiro para retornar o ultimo dia do mês, falso para retornar N* dias (N = configuração)
	 * @return date 
	 */
	public function calculaDataFinal(&$data_inicial, $fimDoMes=FALSE) {
		if ($fimDoMes) {
			list($y, $m, $d) = explode("-", $data_inicial);
			$mk = mktime(0, 0, 0, $m, $d, $y);
			return "$y-$m-" . date("t", $mk);
		}

		$tbConf = new Application_Model_Configuracao();
		$dias = $tbConf->getConfig('AGENDA_MOSTRAR_N_OPCOES');

		list($y, $m, $d) = explode("-", $data_inicial);

		if ((int) "$y$m$d" < (int) date("Ymd")) {
			$data_inicial = date("Y-m-d");
			list($y, $m, $d) = explode("-", $data_inicial);
		}

		$mk = mktime(0, 0, 0, $m, $d + $dias - 1, $y);

		return date("Y-m-d", $mk);
	}

	/**
	 * Diz se pode haver atendimento nesse dia
	 * @param date $data
	 * @param stdClass $sd se o local atende no sábado e domingo
	 * @see Application_Model_Convenio::atendeSabadoEDomingo
	 */
	private function sabadoDomingoOuFeriado($data, $sd) {
		list($y, $m, $d) = explode("-", $data);
		$mk = mktime(0, 0, 0, $m, $d, $y);

		// se não pode fazer no sábado:
		if (!$sd->sabado && date("w", $mk) == 6) {
			return FALSE;
		}

		// se não pode fazer no domingo:
		if (!$sd->domingo && date("w", $mk) == 0) {
			return FALSE;
		}

		$tbFer = new Application_Model_Feriado();
		if ($tbFer->ehFeriado($data)) {
			return FALSE;
		}

		return TRUE;
	}

	/**
	 * Retorna o agendamentos de exame do paciente
	 * Atenção: local do exame: somente med_codigo
	 * @param int $usu_codigo 
	 * @return Zend_Db_Table_Rowset_Abstract
	 */
	public function getHistoricoDeExames($usu_codigo) {
		$where = $this->select(FALSE)
				->setIntegrityCheck(FALSE)
				->from(array("age" => "agenda"), array("age_codigo","age_data_insert","ate_codigo"))
				->join(array("agei" => "agenda_itens"), "agei.age_codigo=age.age_codigo", array("agei_codigo","agei_data","agei_status"))
				->join(array("coni" => "convenio_itens"), "coni.coni_codigo=agei.coni_codigo", array())
				->join(array("conv" => "convenio"), "conv.conv_codigo=coni.conv_codigo", "")
				->join(array("med" => "medico"), "med.med_codigo=conv.med_codigo", array("med_nome"))
				->join(array("proc" => "procedimento"), "proc.proc_codigo=coni.proc_codigo", array("proc_nome"))
				->where("age.usu_codigo=?", $usu_codigo)
				->order("agei_data DESC")
		;

		return $this->fetchAll($where);
	}

}
